package net.zomis.plugin.nonstandard;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import net.zomis.UtilZomisUtils;
import net.zomis.minesweeper.game.MinesweeperField;
import net.zomis.minesweeper.game.MinesweeperMove;
import net.zomis.minesweeper.game.MinesweeperPlayingPlayer;
import net.zomis.minesweeper.weapons.BaseWeapon;

public class SizedBombWeapon extends BaseWeapon {

	private final int bombSize;
	private final int boardSize;
	
	public static final int USE_COUNT = 1;

	protected int hasUsed = 0;
	
	public SizedBombWeapon(int size, int boardSize) {
		this.bombSize = size;
		this.boardSize = boardSize;
	}
	
	public int getUseCount() {
		return this.hasUsed;
	}
	
	@Override
	public String getWeaponType() {
		return MinesweeperMove.STANDARD_BOMB;
	}

	@Override
	public boolean canUseAt(MinesweeperPlayingPlayer user, MinesweeperField field) {
		Collection<MinesweeperField> affected = this.getFieldsAffected(user, field);
		if (affected == null) return false;
		return affected.size() > 0;
	}

	protected MinesweeperField getMove(MinesweeperPlayingPlayer user, MinesweeperField field) {
		int edgeX = (field.getMap().getFieldWidth() - boardSize) / 2;
		int edgeY = (field.getMap().getFieldHeight() - boardSize) / 2;
		// ensureRange(3 + 1, 4, 16 - 3 - 1 - 1)
		int xpos = UtilZomisUtils.ensureRange(edgeX + getRangeDown(), field.getX(), field.getMap().getFieldWidth() - edgeX - getRangeUp() - 1);
		int ypos = UtilZomisUtils.ensureRange(edgeY + getRangeDown(), field.getY(), field.getMap().getFieldHeight() - edgeY - getRangeUp() - 1);
		return field.getMap().getPosition(xpos, ypos);
	}
	
	private int getRangeDown() {
		return (bombSize - 1) / 2;
	}
	private int getRangeUp() {
		return (int) Math.floor(bombSize / 2.0);
	}
	
	@Override
	public Collection<MinesweeperField> getFieldsAffected(MinesweeperPlayingPlayer user, MinesweeperField field) {
		field = getMove(user, field);
		
		ArrayList<MinesweeperField> result = new ArrayList<MinesweeperField>();
		for (MinesweeperField mf : this.getFieldsWithinRange(field, getRangeDown(), getRangeUp())) {
			if (!mf.isClicked()) result.add(mf);
		}
		
		return result;
	}
	private final Collection<MinesweeperField> getFieldsWithinRange(MinesweeperField field, int rangeDown, int rangeUp) {
		List<MinesweeperField> result = new ArrayList<MinesweeperField>();
		
		for (int x = -rangeDown; x <= rangeUp; x++) {
			for (int y = -rangeDown; y <= rangeUp; y++) {
				MinesweeperField mf = field.getMap().getPosition(field.getX() + x, field.getY() + y);
				if (mf != null) result.add(mf);
			}
		}
		
		return result;
	}

	@Override
	public boolean canUse(MinesweeperPlayingPlayer user) {
		if (user.hasMostScore()) return false;
		return this.hasUsed < SizedBombWeapon.USE_COUNT;
	}

	@Override
	public boolean useAt(MinesweeperMove moveInfo) {
		if (!this.canUseAt(moveInfo.getPlayer(), moveInfo.getMovePosition())) return false;
		
//		moveInfo.getMovePosition()
		MinesweeperMove move = moveInfo.getPlayer().createMove(this, this.getMove(moveInfo.getPlayer(), moveInfo.getMovePosition()));
		for (MinesweeperField mf : this.getFieldsAffected(moveInfo.getPlayer(), move.getMovePosition())) {
			this.expanderOpen(mf, move);
		}
		this.hasUsed++;
		moveInfo.getMap().nextTurn();
		return true;
	}

	@Override
	public void resetUsage() {
		this.hasUsed = 0;
	}
	
	@Override
	public String toString() {
		return (this.hasUsed == 0 ? "Unused " : "USED ") + super.toString();
	}

}
