package net.zomis.plugin.mtmt;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import net.zomis.minesweeper.api.Invite;
import net.zomis.minesweeper.api.Minesweeper;
import net.zomis.minesweeper.api.MinesweeperPlayer;
import net.zomis.minesweeper.api.MinesweeperServer;
import net.zomis.minesweeper.events.invites.InviteGameStartingEvent;
import net.zomis.minesweeper.game.GameTag;
import net.zomis.minesweeper.game.GameTag.TagType;
import net.zomis.minesweeper.game.MinesweeperGame;
import net.zomis.minesweeper.game.MinesweeperPlayingPlayer;

public class Jogo {
//	private Jogo[] parents;
//	private Jogo winnerGame;
//	private Jogo loserGame; // Not needed at the moment since match for 3rd place will be removed.
	
	private static final Integer WAIT_FOR_NOBODY = Integer.MIN_VALUE;
	private static final Integer RESULT_WALKOVER = Integer.MIN_VALUE + 1;
	private static final String	GAMETAG	= "mata-mata";
	private static final String	TOURNAMENT_TAG	= "tournament-game";
	
	private final int[] waitingFor = new int[]{ WAIT_FOR_NOBODY, WAIT_FOR_NOBODY };
	private final Map<MinesweeperPlayer, Integer> playersAndResults = new HashMap<MinesweeperPlayer, Integer>();
	private MinesweeperGame game = null;
	private final MataMata mtmt;

	public Jogo(MataMata mtmt, MinesweeperPlayer player1, MinesweeperPlayer player2) {
		this.mtmt = mtmt;
		this.playersAndResults.put(player1, null);
		this.playersAndResults.put(player2, null);
	}

	public Jogo(MataMata mtmt, MinesweeperPlayer player1, int player2) {
		this.mtmt = mtmt;
		this.playersAndResults.put(player1, null);
		this.waitingFor[0] = player2;
	}

	public Jogo(MataMata mtmt, int player1, int player2) {
		this.mtmt = mtmt;
		this.waitingFor[0] = player1;
		this.waitingFor[1] = player2;
	}
	
	public synchronized boolean informGameEnd(MinesweeperGame game) {
		if (game != this.game) return false;
		
		this.playersAndResults.clear();
		for (MinesweeperPlayingPlayer pp : game.getPlayingPlayers()) {
			this.playersAndResults.put(pp.getPlayer(), pp.getScore());
		}
		
		this.getServer().log("Zomis", "Jogo Informed " + this + " about game end: " + game);
		
		this.mtmt.informJogoEnd(this);
		return true;
	}

	public boolean matches(InviteGameStartingEvent event) {
		if (this.hasWaitingFor()) return false;
		if (this.game != null) return false;
		
		List<MinesweeperPlayer> gamePlayers = event.getGame().getPlayers();
		
		Collection<MinesweeperPlayer> a = new HashSet<MinesweeperPlayer>(this.playersAndResults.keySet());
		if (gamePlayers.size() != a.size()) return false;
		for (MinesweeperPlayer pl : a) {
			if (!gamePlayers.contains(pl)) return false;
		}
		
//		a.removeAll(event.getGame().getPlayers());
//		if (!a.isEmpty()) return false;
//		
//		Collection<MinesweeperPlayer> b = event.getGame().getPlayers();
//		b.removeAll(this.playersAndResults.keySet());
//		if (!b.isEmpty()) return false;
		
		this.game = event.getGame();
		this.game.sendChatMessage("This is a game in " + this.mtmt, 0xFF00FF);
		this.tagGame();
		
		return true;
	}

	private void tagGame() {
		this.game.addTag(this.mtmt.hasId() ? new GameTag(TagType.SERVER, game, GAMETAG, this.mtmt.getId()) : null);
		this.game.addTag(new GameTag(TagType.SERVER, game, TOURNAMENT_TAG, this.mtmt.getJogoIndex(this)));
	}

	private boolean hasWaitingFor() {
		for (int i : this.waitingFor)
			if (i > WAIT_FOR_NOBODY) return true;
		return false;
	}

	public void sendInfo(MinesweeperPlayer player) {
		player.sendChatBy(this.toString(player, true), this.getServer().getServerPlayer());
	}
	
	public String toString(MinesweeperPlayer highlight, boolean useDisplayName) {
		String str = "";
		for (Entry<MinesweeperPlayer, Integer> entry : this.playersAndResults.entrySet()) {
			Minesweeper.getServer().log("Mtmt", "toString loop " + this.mtmt.getJogoIndex(this) + ": " + entry.getKey() + ": " + entry.getValue());
			
			if (!str.isEmpty()) str += " vs. ";
			
			if (entry.getKey().equals(highlight)) str += String.format("<font color=\"#ffff00\">");
			
			str += (useDisplayName ? entry.getKey().getDisplayName() : entry.getKey().getName());

			if (entry.getValue() != null) str += " (" + (entry.getValue() == RESULT_WALKOVER ? "WO" : entry.getValue()) + ")";
			
			if (entry.getKey().equals(highlight)) str += "</font>";
			
		}

		for (int i : this.waitingFor) {
			if (i == WAIT_FOR_NOBODY) continue;
			
			if (!str.isEmpty()) str += " vs. ";
			
			Jogo jogo = this.mtmt.getJogo(Math.abs(i));
			
			str += "(" + (i >= 0 ? "Winner" : "Loser") + " of " + jogo.getGameDescription() + ")";
		}
		
		return this.getGameDescription() + ": " + str;
	}
	
	@Override
	public String toString() {
		return this.toString(null, true);
	}
	
	private String getGameDescription() {
		return "Game" + (this.game != null ? "id " + this.game.getGameID() : " " + this.mtmt.getJogoIndex(this));
	}

	public boolean autoAIcheck() {
		if (this.game != null) return false;
		
		for (int i : this.waitingFor) {
			if (i != WAIT_FOR_NOBODY) return false;
		}
		
//		if (this.playersAndResults.isEmpty()) return;
		
		for (Entry<MinesweeperPlayer, Integer> ee : this.playersAndResults.entrySet()) {
			if (!ee.getKey().isAI()) return false;
			if (ee.getValue() != null) return false;
		}
		
//		this.getServer().broadcastChat("Jogo AutoAICheck: " + this);
		
		Invite invt = null;
		for (Entry<MinesweeperPlayer, Integer> ee : this.playersAndResults.entrySet()) {
			if (invt == null) invt = ee.getKey().createInvitation(this.mtmt.getPlugins());
			else {
				invt.sendInvite(ee.getKey());
			}
		}
//		this.getServer().broadcastChat("Starting Invite...");
		
		if (invt == null) return false;
		
		MinesweeperGame g = invt.startGame();
//		this.getServer().broadcastChat("Invite OK: " + g);
		if (g != null) g.setAIDelay(2000);
		
		return g != null;
	}

	private MinesweeperServer getServer() {
		return this.mtmt.getServer();
	}

	public boolean informJogoEnd(Jogo jogo, int index) {
		boolean changed = false;
		for (int i = 0; i < this.waitingFor.length; i++) {
			int value = this.waitingFor[i];
			if (Math.abs(value) != index) continue; // This game is not interesting, NEXT!
			
			this.getServer().log("Zomis", "Jogo inform jogo end " + this + " about : " + jogo + " index " + index + " and value " + value);

			for (MinesweeperPlayer player : jogo.getPlayers()) {
				this.getServer().log("Zomis", "Jogo isWinner " + jogo.isWinner(player) + " isLoser " + jogo.isLoser(player)  + " and value " + value);
				if (jogo.isWinner(player) && value >= 0) {
					this.playersAndResults.put(player, null);
					this.waitingFor[i] = WAIT_FOR_NOBODY;
				}
				else if (jogo.isLoser(player) && value < 0) {
					this.playersAndResults.put(player, null);
					this.waitingFor[i] = WAIT_FOR_NOBODY;
				}
			}
			
			if (this.waitingFor[i] != WAIT_FOR_NOBODY) {
				for (int j = 0; j < jogo.waitingFor.length; j++) {
					if (jogo.waitingFor[j] != WAIT_FOR_NOBODY) {
						this.getServer().log("Zomis", "waitingFor i " + i + " is " + this.waitingFor[i] + " will be set to " + jogo.waitingFor[j]);
						this.waitingFor[i] = jogo.waitingFor[j];
					}
					// a little unsure about this...
				}
			}
			
			changed = true;
		}
		
		if (changed && !this.hasWaitingFor()) {
			this.alertPlayers();
		}
		
		return changed;
	}

	private boolean isLoser(MinesweeperPlayer player) {
		if (this.game != null) {
			if (!this.game.isGameOver()) throw new IllegalStateException("Game is not over: " + this.game + " for jogo " + this);
			MinesweeperPlayingPlayer pp = this.game.getPlayingPlayer(player);
			return pp.isEliminated() && pp.getResultPosition() > this.game.getPlayingPlayers().size() / 2; // check if player is in the lower half of results
		}
		else {
			return this.playersAndResults.get(player) == Jogo.RESULT_WALKOVER;
		}
	}

	private Set<MinesweeperPlayer> getPlayers() {
		if (this.game != null) {
			Set<MinesweeperPlayer> set = new HashSet<MinesweeperPlayer>();
			for (MinesweeperPlayingPlayer pp : this.game.getPlayingPlayers()) {
				set.add(pp.getPlayer());
			}
			return set;
		}
		else {
			return this.playersAndResults.keySet();
		}
	}

	private boolean isWinner(MinesweeperPlayer player) {
		if (this.game != null) {
			if (!this.game.isGameOver()) throw new IllegalStateException("Game is not over: " + this.game + " for jogo " + this);
			MinesweeperPlayingPlayer pp = this.game.getPlayingPlayer(player);
			return pp.isEliminated() && pp.getResultPosition() <= this.game.getPlayingPlayers().size() / 2; // check if player is in the upper half of results
		}
		else {
			if (this.isWalkover()) {
//				if (this.playersAndResults == null) ????
				
				return this.playersAndResults.get(player) != Jogo.RESULT_WALKOVER;
			}
			else throw new IllegalStateException("Jogo is not null and is not walkover: " + this);
		}
	}

	private boolean isWalkover() {
		for (Integer value : this.playersAndResults.values()) {
			if (value != null && value == Jogo.RESULT_WALKOVER) return true;
		}
		return false;
	}

	public MinesweeperGame getGame() {
		return this.game;
	}

	public boolean isFinished() {
		return this.getGame() != null && this.getGame().isGameOver();
	}

	private boolean alerted = false;
	
	public void alertPlayers() {
		if (this.alerted) return;
		if (this.hasWaitingFor()) return;
		if (this.game != null) return;
		
		for (MinesweeperPlayer p : this.playersAndResults.keySet()) {
			p.sendAlert("Mata-Mata:\n" + this.toString(null, false));
		}
		
		this.alerted = true;
	}

	public void informWalkover(MinesweeperPlayer player) {
		if (!this.playersAndResults.containsKey(player)) return;
		
		if (this.game == null) {
			this.playersAndResults.put(player, Jogo.RESULT_WALKOVER);
			this.mtmt.informJogoEnd(this);
		}
		else {
			if (this.game.getPlayingPlayer(player).isEliminated()) return;
			
			this.game.getPlayingPlayer(player).eliminateLoss(); // Let the Jogo informGameEnd method take care of the rest.
			this.mtmt.walkover(player); // Walkover again to make sure that the player is vanished from mtmt.
//			this.mtmt.informJogoEnd(this);
		}
	}

	
//	private void sleep() {
//		try {
//			Thread.sleep(1000);
//		} catch (InterruptedException e) {
//			e.printStackTrace();
//		}
//	}
	
}








