package net.zomis.plugin.editor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import net.zomis.minesweeper.api.FieldInfo;
import net.zomis.minesweeper.api.FieldType;
import net.zomis.minesweeper.game.MinesweeperField;
import net.zomis.minesweeper.game.MinesweeperMove;
import net.zomis.minesweeper.game.MinesweeperPlayingPlayer;
import net.zomis.minesweeper.weapons.BaseWeapon;

public class EditorWeapon extends BaseWeapon {
	private static final int POS_MINE = 9;
	private static final int POS_BLOCKED = 10;
	private static final int POS_UNCLICKED = 11;

	private FieldType selectedFieldType;
	private int selectedValue;
	private boolean selectedIsMine;
	
	private List<MinesweeperField> editorFields = new ArrayList<MinesweeperField>();
	private final MinesweeperPlayingPlayer player;
	private int editorRow;
	
	public EditorWeapon(MinesweeperPlayingPlayer player) {
		this.player = player;
		this.setEditorFields(0);
	}
	
	public void setEditorFields(int row) {
		this.editorRow = row;
		for (MinesweeperField field : this.editorFields) {
			field.sendInfoToPlayer(this.player);
			if (this.player.getPlayer().isClientFlash()) field.sendInfoToPlayer(this.player);
		}
		
		this.editorFields.clear();
		
		for (int i = 0; i < this.player.getMap().getFieldWidth(); i++) {
			MinesweeperField ff = this.player.getMap().getPosition(i, row);
			if (i <= 8)
				this.player.getPlayer().sendFieldInfo(ff, new FieldInfo(i)); // Numbers
			else {
				if (i == POS_MINE) this.player.getPlayer().sendFieldInfo(ff, FieldInfo.VISIBLE_MINE); // Mine
				if (i == POS_BLOCKED) this.player.getPlayer().sendFieldInfo(ff, FieldInfo.BLOCKED); // Blocked
				if (i == POS_UNCLICKED) this.player.getPlayer().sendFieldInfo(ff, FieldInfo.UNCLICKED); // Unclicked
			}
			this.editorFields.add(ff);
		}
		if (row < this.player.getMap().getFieldHeight() / 2) {
			row++;
		}
		else row--;
		
		for (int i = 0; i < this.player.getMap().getFieldWidth(); i++) {
			MinesweeperField ff = this.player.getMap().getPosition(i, row);
			this.player.getPlayer().sendFieldInfo(ff, FieldInfo.BLOCKED); // Blocked
			this.editorFields.add(ff);
		}
	}

	@Override
	public String getWeaponType() {
		return MinesweeperMove.STANDARD_CLICK;
	}

	@Override
	public boolean canUse(MinesweeperPlayingPlayer user) {
		return true;
	}

	@Override
	public boolean canUseAt(MinesweeperPlayingPlayer user, MinesweeperField field) {
		return true;
	}

	@Override
	public Collection<MinesweeperField> getFieldsAffected(MinesweeperPlayingPlayer user, MinesweeperField field) {
		ArrayList<MinesweeperField> result = new ArrayList<MinesweeperField>();
		result.add(field);
		return result;
	}

	@Override
	public boolean useAt(MinesweeperMove moveInfo) {
		if (!this.canUseAt(moveInfo.getPlayer(), moveInfo.getMovePosition())) return false;
		
		if (this.editorFields.contains(moveInfo.getMovePosition()) ||
				this.editorRow == moveInfo.getMovePosition().getY()) {
			if (moveInfo.getMovePosition().getY() != this.editorRow) {
				this.setEditorFields(this.player.getMap().getFieldHeight() - this.editorRow - 1);
			}
			
			this.selectedValue = moveInfo.getMovePosition().getX();
			if (this.selectedValue > 8) this.selectedValue = -1;
			
			this.selectedIsMine = moveInfo.getMovePosition().getX() == POS_MINE;
			
			this.selectedFieldType = moveInfo.getMovePosition().getX() == POS_BLOCKED ? FieldType.BLOCKED : 
				moveInfo.getMovePosition().getX() == POS_UNCLICKED ? FieldType.UNCLICKED : FieldType.CLICKED;
			
			this.player.getPlayer().sendChat("Changed selection: " + String.format("%d %s %d", this.selectedValue, this.selectedIsMine + "", this.selectedFieldType));
			
			return true;
		}
		
		
		for (MinesweeperField field : this.getFieldsAffected(moveInfo.getPlayer(), moveInfo.getMovePosition())) {
			field.setBlocked(this.selectedFieldType == FieldType.BLOCKED);
			if (this.selectedFieldType == FieldType.UNCLICKED) {
				field.inactivate();
			}
			else field.setMine(this.selectedIsMine);

			if (this.selectedValue >= 0) {
				field.setValue(this.selectedValue);
				field.activate();
			}
			if (this.selectedFieldType != FieldType.UNCLICKED) {
				field.activate(moveInfo);
			}
			
			field.sendInfoToPlayers();
		}
		
		return true;
	}

	@Override
	public void resetUsage() {
		this.setEditorFields(0);
	}

}
