package net.zomis.commands;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;

import net.zomis.UtilZomisList;
import net.zomis.UtilZomisUtils;
import net.zomis.commands.GameResults1v1.PlayerResults;
import net.zomis.minesweeper.api.Invite;
import net.zomis.minesweeper.api.Minesweeper;
import net.zomis.minesweeper.api.MinesweeperPlayer;
import net.zomis.minesweeper.api.MinesweeperPlugin;
import net.zomis.minesweeper.api.MinesweeperServer;
import net.zomis.minesweeper.events.Command;
import net.zomis.minesweeper.events.Event;
import net.zomis.minesweeper.events.EventListener;
import net.zomis.minesweeper.events.invites.InviteGameEndedEvent;
import net.zomis.minesweeper.events.player.PlayerCommandEvent;
import net.zomis.minesweeper.game.MinesweeperGame;
import net.zomis.minesweeper.game.MinesweeperMove;
import net.zomis.minesweeper.game.MinesweeperPlayingPlayer;
import net.zomis.plugin.classic.PluginClassicGame;
import net.zomis.utils.ZomisUtils;

public class AIsCommands extends MinesweeperPlugin implements EventListener {
	private boolean aiRatingFix = false;
	private int aiRatingFixGames = 0;
	
	Collection<MinesweeperPlayer> aiScreenSavers = new HashSet<MinesweeperPlayer>();
	
	@Command(command = "aiscreen", help = "AI Screensaver", requiredPermission = PlayerCommandEvent.PERMISSION_DONATOR)
	public void aiObserveScreensaver(PlayerCommandEvent event) {
		if (this.aiScreenSavers.contains(event.getPlayer()))
			this.aiScreenSavers.remove(event.getPlayer());
		else {
			this.aiScreenSavers.add(event.getPlayer());
			
			Collection<MinesweeperPlayer> set = new HashSet<MinesweeperPlayer>();
			set.add(event.getPlayer());
			this.addAIScreenSaverGame(event.getPlayer().getServer(), set);
		}
	}
	
	private void addAIScreenSaverGame(MinesweeperServer server, Collection<MinesweeperPlayer> observers) {
		MinesweeperPlayer p1 = UtilZomisList.getRandom(server.getAIs());
		MinesweeperPlayer p2 = UtilZomisList.getRandom(server.getAIs());
		Invite invt = p1.createInvitation(null);
		invt.sendInvite(p2);
		invt.setDatabase(false);
		invt.setPublic(true);
		invt.setAIDelay(true);
		MinesweeperGame game = invt.startGame();
		for (MinesweeperPlayer player : observers)
			game.addObserver(player);
	}

	@Event
	public void aiObserveScreensaverGameEnd(InviteGameEndedEvent event) {
		Collection<MinesweeperPlayer> inThisGame = new HashSet<MinesweeperPlayer>();
		for (MinesweeperPlayer aiScreensaver : aiScreenSavers) {
			if (event.getGame().getObservers().contains(aiScreensaver)) {
				event.getGame().removeObserver(aiScreensaver);
				inThisGame.add(aiScreensaver);
			}
		}
		if (!inThisGame.isEmpty())
			this.addAIScreenSaverGame(event.getGame().getServer(), inThisGame);
	}
	
	@Command(command = "airatingfix", help = "Match AIs against each other continously.", requiredPermission = PlayerCommandEvent.PERMISSION_ADMIN)
	public void onAIratingFix(PlayerCommandEvent event) {
		if (event.getServer().isAIRatingsLocked()) {
			event.getPlayer().sendChat("AI Ratings is locked. Use /ailock to unlock it.");
			return;
		}
		
		this.aiRatingFix = !this.aiRatingFix;
		if (this.aiRatingFix) this.aiRatingFixGames = 0;
		event.getPlayer().getServer().threadDisableLogging(event);
		
		while (this.aiRatingFix) {
			List<MinesweeperPlayer> ais = event.getPlayer().getServer().getAIs();
			MinesweeperPlayer oneAI = UtilZomisList.getRandom(ais);
			ais.remove(oneAI);
			MinesweeperPlayer anotherAI = UtilZomisList.getRandom(ais);
			MinesweeperGame game = event.getPlayer().getServer().createAITestGame(null, new String[]{ oneAI.getName(), anotherAI.getName() });
			
			if (game == null) continue;
			int clicks = 0;
			int CLICKS_TIMEOUT = 1000;
			while (!game.isGameOver() && clicks < CLICKS_TIMEOUT) {
				game.callAI();
				clicks++;
			}
//			Thread.sleep(500);
			this.aiRatingFixGames++;
		}
		event.getPlayer().sendChat("End of AI Rating Games." + this.aiRatingFixGames);
		
	}
	
	@Command(command = "aimoves", help = "Find out where all the AIs would play in the current game.", requiredPermission = PlayerCommandEvent.PERMISSION_TRUSTED_USER)
	public void onAImoves(PlayerCommandEvent event) {
		MinesweeperGame game = getGameForEvent(event);
		String aiName = event.getParameter(0);
		if (!aiName.startsWith("#")) aiName = event.getParameter(1);
		if (!aiName.startsWith("#")) aiName = null;
		
		for (MinesweeperPlayer ai : game.getServer().getAIs()) {
			if (aiName != null) {
				if (!ai.getName().equalsIgnoreCase(aiName)) continue;
			}
			
			MinesweeperMove move = null;
			long time = System.nanoTime();
			move = game.getMapFactory().ai(game.getCurrentPlayer(), game.getServer().getAI(ai.getName())).play();
			time = System.nanoTime() - time;
			event.getPlayer().sendChat(ai.getName() + " move: " + move.getWeaponType() + " @ " + move.getMovePosition().getCoordinate() + " (" + ZomisUtils.nanoToMilli(time) + " ms)");
		}
	}
	
	private int aiProgress = -1;
	private int aisCount = -1;
	private GameResults1v1 results;
	private long aisCommandStart;
	private String	plugin;
	
	
	@Command(command = "ais_plugin", help = "Change the plugin for the /ais command", requiredPermission = PlayerCommandEvent.PERMISSION_TRUSTED_USER)
	public void onAIsPlugin(PlayerCommandEvent event) {
		this.plugin = event.getParameter(0);
	}
	@Command(command = "ais", help = "See how the AIs perform against each other", requiredPermission = PlayerCommandEvent.PERMISSION_TRUSTED_USER)
	public void onAIs(PlayerCommandEvent event) throws InterruptedException {
		if (plugin == null) {
			plugin = PluginClassicGame.class.getSimpleName();
		}
		
		if (aiProgress != -1) {
			event.getPlayer().sendChat("There is currently an operation in progress. Processing game " + aiProgress + " / " + aisCount);
			
			if (event.getParameter(0).contentEquals("STOP")) {
				event.getPlayer().sendChat("Stopping the current AI testing. Please wait for the current game to finish.");
				aisCount = aiProgress;
			}
			if (results != null) {
				this.logAIwins(event.getServer());
			}
			return;
		}
		if (results != null) {
			this.logAIwins(event.getServer());
		}
		
		aisCommandStart = System.nanoTime();
		
		String countString = event.getParameter(0);
		try {
			aisCount = Integer.parseInt(countString);
		}
		catch (NumberFormatException e) {
			aisCount = 500;
		}
		int MAX = 1000000;
		int UPDATE_PERIOD = 100;
		boolean verbose = (aisCount <= 100);
		if (aisCount > MAX) aisCount = MAX;
		
		List<MinesweeperPlayer> ais;
		
		// Check if user has defined AIs too
		if (event.getParameters().length > 2) {
			ais = new ArrayList<MinesweeperPlayer>();
			for (int i = 1; i < event.getParameters().length; i++) {
				MinesweeperPlayer player = event.getPlayer().getServer().getPlayerByName(event.getParameter(i));
				if (player != null && player.isAI()) ais.add(player);
			}
		}
		else {
			ais = event.getPlayer().getServer().getAIs();
			
			// Remove some AIs that should not participate
			Iterator<MinesweeperPlayer> it = ais.iterator();
			while (it.hasNext()) {
				MinesweeperPlayer p = it.next();
				if (p.getName().contentEquals("#AI_OpenField")) it.remove();			// is way too much cheat in another way.
				else if (p.getName().contentEquals("#AI_Impossible")) it.remove();		// is way too much cheat.
				else if (p.getName().contentEquals("#AI_Loser")) it.remove();			// is way too slow.
				else if (p.getName().contentEquals("#AI_Complete_Idiot")) it.remove();	// is way too stupid.
			}
		}
		
		if (ais.size() <= 1) {
			event.getPlayer().sendChat("Too few AIs specified.");
			return;
		}
		else {
			boolean hasUniqueAIs = new HashSet<MinesweeperPlayer>(ais).size() == ais.size();
			results = new GameResults1v1(!hasUniqueAIs);
			event.getPlayer().getServer().broadcastChat("Starting " + aisCount + " test games!");
			event.getPlayer().getServer().threadDisableLogging(event);
			for (MinesweeperPlayer ai : ais) {
				event.getPlayer().getServer().broadcastChat("Test AI: " + ai.getDisplayName());
			}
		}
		
		for (int i = 1; i <= aisCount; i++) {
			if (i % UPDATE_PERIOD == 0) event.getPlayer().getServer().broadcastChat("Processing game " + i + " / " + aisCount); 
			aiProgress = i;
			MinesweeperPlayer[] players = new MinesweeperPlayer[2];
			
			// Randomize two AIs that should fight against each other.
			ArrayList<MinesweeperPlayer> randomAIs = new ArrayList<MinesweeperPlayer>(ais);
			players[0] = UtilZomisList.getRandom(randomAIs);
			randomAIs.remove(players[0]);
			players[1] = UtilZomisList.getRandom(randomAIs);
			
			MinesweeperGame game = event.getPlayer().getServer().createAITestGame(new String[]{ this.plugin }, new String[]{ players[0].getName(), players[1].getName() });
			if (game == null) {
				if (verbose) event.getPlayer().getServer().broadcastChat("Unable to start game: " + players[0].getName() + " vs. " + players[1].getName());
				i--; // Do not count this game.
				continue;
			}
			List<MinesweeperPlayingPlayer> pplayers = game.getPlayingPlayers(); // Store here because they are removed when game is finished.
			
			int clicks = 0;
			int CLICKS_TIMEOUT = 1000;
			while (!game.isGameOver() && clicks < CLICKS_TIMEOUT) {
				game.callAI();
				clicks++;
			}
			
			if (clicks >= CLICKS_TIMEOUT) {
				event.getPlayer().getServer().broadcastChat("Infinite loop: " + game); // Always do this, because it is a serious error.
			}
			else {
				results.saveResult(pplayers);
			}
			
			if (verbose) {
				event.getPlayer().getServer().broadcastChat("Test game " + i + ": " + UtilZomisUtils.implode(", ", pplayers));
				Thread.sleep(1000);
			}
			
			pplayers.clear();
			game.endGame();
		}
		this.logAIwins(event.getServer());
		aiProgress = -1;
	}
	
	private void logAIwins(MinesweeperServer server) {
		if (this.results == null) return;
		if (this.aiProgress == 0) return;
		long timeTaken = System.nanoTime() - aisCommandStart;
		double timeMS = UtilZomisUtils.nanoToMilli(timeTaken);
		double gamesPerSecond = this.aiProgress / (timeMS / 1000.0);
		
		server.broadcastLobbyChat(this.aiProgress + " test games finished: " + timeMS + " ms. " + gamesPerSecond + " games per second.");
		
		for (Entry<String, PlayerResults> presult : results.getResults().entrySet()) {
			server.broadcastLobbyChat("Results for " + presult.getKey() + ":");
			PlayerResults results = presult.getValue();
			for (String key : results.getKeys()) {
				server.broadcastLobbyChat(String.format("Opponent %s: %d / %d (%.4f)", key, results.getWins(key), results.getTotal(key), results.getPercent(key)));
			}
			server.broadcastLobbyChat("");
		}
	}
	
	private MinesweeperGame getGameForEvent(PlayerCommandEvent event) {
		MinesweeperGame game = null;
		if (event.getParameter(0) != null) {
			try {
				game = event.getPlayer().getServer().getGame(Integer.parseInt(event.getParameter(0)));
			}
			catch (NumberFormatException e) {
//				event.getPlayer().sendChat(event.getCommand() + ": No valid game parameter specified for command '/" + event.getCommand() + "', using the current game.");
			}
		}
		if (game == null) game = event.getPlayer().getGame();
		return game;
	}
	
	@Command(command = "ai", help = "Start a game with only AIs playing, and observe it", requiredPermission = PlayerCommandEvent.PERMISSION_DONATOR)
	public void onAIgame(PlayerCommandEvent event) {
		Invite invt;
		MinesweeperPlayer playerA = event.getPlayer().getServer().getPlayerByName(event.getParameter(0));
		if (playerA != null && !playerA.isAI() && !playerA.equals(event.getPlayer())) {
			event.getPlayer().sendChat("You are not allowed to do that.");
			return;
		}
		
		
		if (event.getParameters().length >= 2) {
			invt = playerA.createInvitation(null);
			if (invt == null) {
				event.getPlayer().sendChat("Could not create invitation for player " + event.getParameter(0));
			}
			
			for (int i = 1; i < event.getParameters().length; i++) {
				MinesweeperPlayer player = event.getPlayer().getServer().getPlayerByName(event.getParameter(i));
				
				if (playerA != null && !playerA.isAI() && !playerA.equals(event.getPlayer())) {
					invt.cancel();
					event.getPlayer().sendChat("You are not allowed to do that.");
					return;
				}
				invt.sendInvite(player);
			}
			if (invt.getPlayers().size() != event.getParameters().length) {
				event.getPlayer().sendChat("Unable to start game. Not enough players.");
				return;
			}
		}
		else {
			List<MinesweeperPlayer> ais = event.getPlayer().getServer().getAIs();
			MinesweeperPlayer pl1 = UtilZomisList.getRandom(ais);
			MinesweeperPlayer pl2;
			do {
				pl2 = UtilZomisList.getRandom(ais);
			}
			while (pl1 == pl2);
			
			invt = event.getPlayer().getServer().getPlayerByName(pl1.getName()).createInvitation(null);
			if (invt == null) {
				event.getPlayer().sendChat("Could not create invitation for player " + pl1);
			}
			
			invt.sendInvite(pl2.getName());
			if (invt.getPlayers().size() != 2) {
				event.getPlayer().sendChat("Unable to start game. " + pl1 + ", " + pl2);
				return;
			}
		}
		
		boolean isOnlyAIs = true;
		for (MinesweeperPlayer pp : invt.getPlayers()) {
			if (!pp.isAI()) isOnlyAIs = false;
		}
		
		if (isOnlyAIs) invt.setDatabase(false);
		MinesweeperGame game = invt.startGame();
		if (!event.getPlayer().isInGame() && game != null) game.addObserver(event.getPlayer());
	}

	@Override
	public void onEnable() {
		this.registerListener(this);
	}

	@Override
	public void onDisable() {
		this.logAIwins(Minesweeper.getServer());
		this.aisCount = this.aiProgress;
	}

	@Override
	public boolean canBeChosenBy(MinesweeperPlayer user) {
		return false;
	}

}
