package net.zomis.minesweeper.game.model;

import java.util.Random;

import net.zomis.minesweeper.api.ai.AIHelper;
import net.zomis.minesweeper.api.MapFactory;
import net.zomis.minesweeper.api.ai.AISupplier;
import net.zomis.minesweeper.api.ai.MinesweeperAI;
import net.zomis.minesweeper.api.MinesweeperPlayer;
import net.zomis.minesweeper.api.MinesweeperPlugin;
import net.zomis.minesweeper.events.EventRegistrator;
import net.zomis.minesweeper.events.game.GameInitEvent;
import net.zomis.minesweeper.functional.Supplier;
import net.zomis.minesweeper.game.MinesweeperField;
import net.zomis.minesweeper.game.MinesweeperMap;
import net.zomis.minesweeper.game.MinesweeperPlayingPlayer;
import net.zomis.minesweeper.weapons.MinesweeperWeapon;
import net.zomis.minesweeper.weapons.classic.BombWeapon;
import net.zomis.minesweeper.weapons.classic.ClickWeapon;

public class MapFactoryImpl implements MapFactory {
	public static final int	DEFAULT_PLAYER_COUNT = 2;
	
	private static final Random random = new Random();
	
	private MinesweeperMap map;
	private AIHelper aiHelper;
	private boolean	loaded = false;

	@Override
	public MapFactory setAIHelper(AIHelper aiHelper) {
		this.aiHelper = aiHelper;
		return this;
	}

	private MapFactoryImpl() {
		this.map = newStandardSizeAndNeighbors();
	}
	
	private MapFactoryImpl(int x, int y) {
		this.map = new MfeMapImpl();
		((MfeMap) map).initMap(x, y);
	}
	
	private MapFactoryImpl(MinesweeperMap map) {
		this.map = map;
	}
	
	@Deprecated
	private static MinesweeperMap newStandardSizeAndNeighbors() {
		MfeMap map = new MfeMapImpl();
		map.initMap(16, 16);
		
		for (MinesweeperField ff : map) {
			for (int xx = -1; xx <= 1; xx++)
				for (int yy = -1; yy <= 1; yy++)
					if (xx != 0 || yy != 0)
						ff.addNeighbor(xx, yy);
		}
		return map;
	}

	@Override
	public MinesweeperMap map() {
		map.executeEvent(new GameInitEvent(map));

		if (!this.loaded) {
			int mcount = map.getMinesCount();
			map.generate(mcount, random);
		}

		return this.map;
	}
	
	public static MapFactory newFactory() {
		return new MapFactoryImpl();
	}
	public static MapFactory fromMap(MinesweeperMap map) {
		return new MapFactoryImpl(map);
	}

	@Override
	public MapFactory withDefaultWeapons() {
		return withWeapon(ClickWeapon.supplier).withWeapon(BombWeapon.supplier);
	}

	@Override
	public MinesweeperMap standard() {
		this.withPlayers(DEFAULT_PLAYER_COUNT - this.map.getPlayingPlayers().size());
		this.withWeapon(new Supplier<MinesweeperWeapon>() {
            @Override
            public MinesweeperWeapon get() {
                return new ClickWeapon();
            }
        });
        this.withWeapon(new Supplier<MinesweeperWeapon>() {
            @Override
            public MinesweeperWeapon get() {
                return new BombWeapon();
            }
        });
		return this.map();
	}

    @Override
    public MapFactory withWeapon(Supplier<? extends MinesweeperWeapon> weapon) {
		if (this.map.getPlayingPlayers().isEmpty()) throw new IllegalStateException("Players must be added before adding weapons");
		
		for (MinesweeperPlayingPlayer pp : map.getPlayingPlayers()) {
            MinesweeperWeapon weap = weapon.get();
            pp.giveWeapon(weap);
		}
		return this;
	}

	@Override
	public MinesweeperMap hiddenMap(MinesweeperMap map) {
		HiddenMap hidden = new HiddenMap(map);
		hidden.initMap(map.getFieldWidth(), map.getFieldHeight());
		hidden.loadMap(map.saveMap());
		return hidden;
	}

	@Override
	public MapFactory loadFrom(String str) {
		if (this.map.getPlayingPlayers().isEmpty()) throw new IllegalStateException("Map must have players before it can be loaded from string");
		this.loaded = true;
		this.map.generate();
		this.map.loadMap(str);
		return this;
	}

	@Override
	public MapFactory withTwoPlayers() {
		return this.withPlayers(2);
	}

	@Override
	public MapFactory withPlayers(int count) {
		for (int i = 0; i < count; i++)
			((MfeMap) map).addPlayer(new MfePlayer(map, null, map.getPlayingPlayers().size()));
		return this;
	}

    @Override
	public MapFactory withAI(AISupplier aiSupplier, MinesweeperPlayer backingPlayer) {
		if (aiSupplier == null) {
            throw new IllegalArgumentException("AI supplier cannot be null");
        }
        ((MfeMap) map).addPlayer(new MfeAI(map, backingPlayer,
                aiSupplier,
                map.getPlayingPlayers().size()));
		return this;
	}
	
	@Override
    public MinesweeperAI ai(MinesweeperPlayingPlayer player, AISupplier aiSupplier) {
		if (aiSupplier == null) {
            throw new IllegalArgumentException("AI supplier cannot be null");
        }
        return aiSupplier.get(player);
	}
	
	@Override
	public MapFactory withSize(int width, int height) {
		return new MapFactoryImpl(width, height);
	}
	
	public static MapFactory cleanFactory() {
		return new MapFactoryImpl(16, 16);
	}
	
	@Override
	public MapFactory withPlugin(MinesweeperPlugin plugin) {
		this.map.addPlugin(plugin);
		return this;
	}

	@Override
	public MapFactory withAI(AISupplier aiClass) {
		return this.withAI(aiClass, null);
	}

	@Override
	public MapFactory withPlayer(MinesweeperPlayer player) {
		((MfeMap) this.map).addPlayer(new MfePlayer(this.map, player, this.map.getPlayingPlayers().size()));
		return this;
	}

	@Override
	public MapFactory withEvents(EventRegistrator events) {
		map.registerEvents(events);
		return this;
	}

}
