package net.zomis.minesweeper.game.model;

import net.zomis.minesweeper.game.MinesweeperField;
import net.zomis.minesweeper.game.MinesweeperMap;
import net.zomis.minesweeper.game.MinesweeperMove;
import net.zomis.minesweeper.game.MinesweeperPlayingPlayer;
import net.zomis.minesweeper.game.MoveAllowedState;
import net.zomis.minesweeper.weapons.MinesweeperWeapon;

public class MfeMove implements MinesweeperMove {
	private final MinesweeperWeapon weapon;
	private final MinesweeperField movePosition;
	private final MinesweeperPlayingPlayer player;
	private final String weaponType;
	private final MinesweeperMap	map;
	/**
	 * @return The {@link MinesweeperWeapon}, if any, used for this move.
	 */
	public MinesweeperWeapon getWeapon() {
		return this.weapon;
	}
	/**
	 * @return The type of {@link MinesweeperWeapon}, if any, used for this move.
	 */
	public String getWeaponType() {
		if (this.weapon == null) return this.weaponType;
		return this.weapon.getWeaponType();
	}
	/**
	 * @return The {@link MinesweeperMap} on which the move took place
	 */
	public MinesweeperMap getMap() {
		return this.map;
	}
	
	protected MfeMove(MinesweeperPlayingPlayer player, String weaponType, MinesweeperField field) {
		this.movePosition = field;
		this.weapon = null;
		this.weaponType = weaponType;
		this.player = player;
		this.map = (field != null ? field.getMap() : player.getMap());
	}
	public MfeMove(MinesweeperPlayingPlayer player, MinesweeperWeapon weapon, MinesweeperField field) {
		this.movePosition = field;
		this.weapon = weapon;
		this.weaponType = (this.weapon == null ? null : this.weapon.getWeaponType());
		this.player = player;
		this.map = (field != null ? field.getMap() : player.getMap());
	}
	/**
	 * @return The {@link MinesweeperField}, if any, at which the move took place
	 */
	public MinesweeperField getMovePosition() {
		return this.movePosition;
	}
	
	/**
	 * Determines whether the move is allowed or not. Based upon if the weapon can be used in the way specified by this object.
	 * 
	 * @return The allowed state of the move
	 */
	public MoveAllowedState getMoveAllowedState() {
		if (player == null) return MoveAllowedState.NO_PLAYER;
		// A check if player actually has the weapon is done later in this file. Perhaps it should be done here already ?
		// TODO: Return not OK if the player is eliminated already
		if (this.getWeapon() == null) {
			return MoveAllowedState.NO_WEAPON;
		}
		if (!this.getMap().isActionMode() && !this.getPlayer().isMyTurn()) {
			return MoveAllowedState.NOT_PLAYER_TURN;
		}
		if (!this.getWeapon().canUse(player)) {
			return MoveAllowedState.WEAPON_DENIED;
		}
		// TODO: Weapon/Move allowance, should there be a specific move position != null or not? Considering weapons that does not need a MovePosition.
		if (this.getMovePosition() != null && !this.getWeapon().canUseAt(this.getPlayer(), this.getMovePosition())) {
			return MoveAllowedState.WEAPON_POSITION_DENIED;
		}
		if (!this.getWeapon().canUseAt(this.getPlayer(), this.getMovePosition())) {
			return MoveAllowedState.WEAPON_POSITION_DENIED;
		}
		else return MoveAllowedState.OK;
	}
	
	
	/**
	 * Convenience method for getting the X-position of the move position 
	 * @return The X-position of the move associated with this object, or Integer.MIN_VALUE if no position has been specified.
	 * @see #getMovePosition()
	 */
	public int getX() {
		return (this.movePosition != null ? movePosition.getX() : Integer.MIN_VALUE);
	}
	/**
	 * Convenience method for getting the Y-position of the move position 
	 * @return The Y-position of the move associated with this object, or Integer.MIN_VALUE if no position has been specified.
	 * @see #getMovePosition()
	 */
	public int getY() {
		return (this.movePosition != null ? movePosition.getY() : Integer.MIN_VALUE);
	}
	/**
	 * @return The player associated with this move
	 */
	public MinesweeperPlayingPlayer getPlayer() {
		return this.player;
	}
	
	@Override
	public String toString() {
		return "(Move@" + (this.movePosition != null ? this.movePosition.getCoordinate() : null) + " : " + this.getWeaponType() + " by " + this.getPlayer() + ")";
	}
	
	@Override
	public boolean equals(Object obj) {
		if (!(obj instanceof MinesweeperMove)) return false;
		
		MinesweeperMove move = (MinesweeperMove) obj;
		return move.getPlayer().getIndex() == this.getPlayer().getIndex()
				&& move.getMovePosition() == null ? this.getMovePosition() == null : move.getMovePosition().equals(this.getMovePosition())
				&& move.getWeaponType().contentEquals(this.getWeaponType());
	}
	
	@Override
	public int hashCode() {
		int i = 0;
		i = 103 * i + this.getPlayer().getIndex();
		i = 103 * i + this.getWeaponType().hashCode();
		i = 103 * i + this.getX();
		i = 103 * i + this.getY();
		return i;
	}
	public String getMoveString() {
		return this.getWeaponType() + this.getMovePosition().getCoordinate();
	}
	@Override
	public MinesweeperField getField() {
		return this.getMovePosition();
	}
	
}
