package net.zomis.minesweeper.game.model;

import java.util.*;

import net.zomis.UtilZomisList;
import net.zomis.UtilZomisUtils;
import net.zomis.minesweeper.api.MapFactory;
import net.zomis.minesweeper.api.ai.MinesweeperAI;
import net.zomis.minesweeper.api.MinesweeperPlayer;
import net.zomis.minesweeper.api.MinesweeperPlugin;
import net.zomis.minesweeper.events.BaseEvent;
import net.zomis.minesweeper.events.CancellableEvent;
import net.zomis.minesweeper.events.game.*;
import net.zomis.minesweeper.game.MinesweeperField;
import net.zomis.minesweeper.game.MinesweeperMap;
import net.zomis.minesweeper.game.MinesweeperMove;
import net.zomis.minesweeper.game.MinesweeperPlayingPlayer;
import net.zomis.minesweeper.game.MinesweeperReplay;
import net.zomis.minesweeper.game.MoveAllowedState;
import net.zomis.minesweeper.weapons.MinesweeperWeapon;

public abstract class MfeMap implements MinesweeperMap {
    protected MinesweeperEvents events;

	private static final int POSITION_RADIX = 16;
	private int endGameWhenPlayersLeftCount = 1;
	
	private final Random random = new Random();
	
	protected final List<MinesweeperPlayingPlayer> playingPlayers;
	private final List<MinesweeperPlugin> activePlugins;
    private int mapWidth;
    private int mapHeight;
    private List<List<MinesweeperField>> map;

    public MfeMap() {
		this.playingPlayers = new ArrayList<MinesweeperPlayingPlayer>(2);
		this.moveHistory = new LinkedList<MinesweeperMove>();
		this.actionMode = false;
		this.activePlugins = new LinkedList<MinesweeperPlugin>();
	}
	
	public void addPlayer(MinesweeperPlayingPlayer playing) {
		this.playingPlayers.add(playing);
	}
	
	protected final List<MinesweeperMove> moveHistory;
	private boolean actionMode;
	
	private int playerTurn = 0;
	private int clicks = 0;
	protected boolean serverControlled = false; // if this is true then server and only server will know everything about the board.
	protected boolean endGameCalled = false;
	
	public List<MinesweeperMove> getMoveHistory() {
		return new ArrayList<MinesweeperMove>(this.moveHistory);
	}

	public boolean isServerControlled() {
		return this.serverControlled;
	}
	
	public int getClickCount() {
		return this.clicks;
	}

	private List<MinesweeperField> getAllPlaceableMineFields() {
		List<MinesweeperField> allFields = new ArrayList<MinesweeperField>();
		for (MinesweeperField mf : this) {
			if (mf.isBlocked()) continue;
			allFields.add(mf);
		}
		return allFields;
	}
	
	@Override
	public void generate(int mines) {
		this.generate(mines, staticRandom);
	}
	
	@Override
	public CustomEvent executeCustomEvent(CustomEvent customEvent) {
		return executeEvent(customEvent);
	}
	
	@Override
	public void generate(int mines, Random randomizer) {
		if (randomizer == null) randomizer = random;
		
		executeEvent(new GamePreGenerateEvent(this));
		
		List<MinesweeperField> fields = this.getAllPlaceableMineFields();
		if (fields.size() < mines) {
			throw new IllegalStateException("Not enough room for all mines: " + fields.size() + " vs " + this.getMinesCount());
		}
		this.clearMap();
		this.setCurrentPlayerTurn(0);
		this.moveHistory.clear();
		
		for (int i = 0; i < mines; i++) {
			MinesweeperField random = UtilZomisList.getRandom(fields, randomizer);
			random.setMine(true);
			fields.remove(random);
		}

		this.postGenerate();
	}
	
	
	@Override
	@Deprecated
	public void generate(Random randomizer) {
		this.generate(this.getMinesCount(), randomizer);
	}

	private static Random staticRandom = new Random();
	
	@Override
	public void generate() {
		generate(staticRandom);
	}
	
	private void clearMap() {
		for (MinesweeperField mf : this) {
			if (mf.isClicked()) mf.inactivate();
			mf.setMine(false);
		}
	}

	protected void postGenerate() {
		this.initFields();
		this.clicks = 0;
		
		executeEvent(new GamePostGenerateEvent(this));
	}

	@Override
	public void initFields() {
		for (MinesweeperField mf : this)
			mf.init();
	}
	
	public boolean initMap(int mapWidth, int mapHeight) {
		this.minesCount = (int) (mapWidth * mapHeight * 0.2);

        this.mapWidth = mapWidth;
        this.mapHeight = mapHeight;
        this.map = new ArrayList<List<MinesweeperField>>(mapWidth);
        for (int xx = 0; xx < mapWidth; xx++) {
            this.map.add(xx, new ArrayList<MinesweeperField>(mapHeight));
        }

        for (int yy = 0; yy < mapHeight; yy++) {
            for (int xx = 0; xx < mapWidth; xx++) {
                MinesweeperField t = newTile(this, xx, yy);
                map.get(xx).add(yy, t); // [yy] = t;
            }
        }
        return true;
	}
	
	public void performScoreChange(MinesweeperPlayingPlayer player, int change) {
		player.changeScore(change);
		
//		if (change == 0) return; // This is not what I call a "change" exactly...
//		int old = player.getScore();
//		player.changeScore(change);
//		MinesweeperEvents.executeEvent(new GameScoreChanged(this, player, old));
	}
	
	protected void postMove(MinesweeperMove move) {
		// Move is null when an AI fails to play.
		
		if (move != null) {
			this.clicks++;
			this.moveHistory.add(move);
		}
		
		if (this.endCheckAndPerform()) { // TODO: Shouldn't move event be executed before endCheckAndPerform is called? 
			return;
		}
		if (move != null) {
            executeEvent(new PlayerAfterMoveEvent(move));
        }
	}
	
	protected void removePlayer(MinesweeperPlayingPlayer pp) {
		// this method can be overriden
	}

	protected boolean isReplay = false;
	
	private boolean isAIDelay = false;

	private int	minesCount;
	
	public void setMinesCount(int minesCount) {
		this.minesCount = minesCount;
	}
	
	@Override
	public synchronized void callAI() {
//		logger.debug("Call AI: " + this.getCurrentPlayer() + " isAIDelay? " + this.isAIDelay + " isReplay? " + this.isReplay);
		if (this.isAIDelay) return;
		if (this.isReplay) return;
		
		this.isAIDelay = true;
		if (this.getCurrentPlayer().isEliminated()) return; // AI is eliminated, then it's no use in calling it. (Prevent IllegalStateException when AI propose DRAW)
		
		performAIMove();
	}
	
	protected void sendSound(MinesweeperMove move) {
	}

	private synchronized void performAIMove() {
		isAIDelay = false;
		if (this.isReplay) return;
		if (this.isGameOver()) return;
		
		if (getCurrentPlayer() == null) {
			return;
		}
		
		if (getCurrentPlayer().isEliminated()) {
			this.nextTurn();
			return;
		}
		
		if (getCurrentPlayer().isAI() && !this.endCheck()) {
			MinesweeperPlayingPlayer playingPlayer = getCurrentPlayer();
			
			MinesweeperAI theAI = ((MfeAI) getCurrentPlayer()).getAI();
			boolean moveResult = false;
			MinesweeperMove move = null;
			try {
				move = theAI.play(playingPlayer);
				
				if (!playingPlayer.isEliminated()) {
					if (move != null) {
						if (move.getPlayer() != playingPlayer) {
                            throw new IllegalStateException("Move player does not equals AI player.");
                        }
						moveResult = this.performMove(move).isOK();
					}
				}
				else {
					moveResult = true;
				}
			}
			catch (Exception e) {
				executeEvent(new AIExceptionEvent(theAI, playingPlayer, move, e, this));
				this.postMove(null);
			}
			catch (AssertionError e) {
				executeEvent(new AIExceptionEvent(theAI, playingPlayer, move, e, this));
				this.postMove(null);
			}
			if (!moveResult) {
				executeEvent(new AIExceptionEvent(theAI, playingPlayer, move, null, this));
				((MfePlayer)playingPlayer).surrender();
				this.postMove(null);
			}
		}
		
		// Get current player has changed here, so a null-check is needed.
		if (getCurrentPlayer() != null && getCurrentPlayer().isAI() && !this.endCheck()) {
//			this.internalCallAI();
		}
	}

	public int getFieldWidth() {
		return this.mapWidth;
	}

	public int getFieldHeight() {
		return this.mapHeight;
	}

	public MinesweeperField getPosition(int x, int y) {
        if (map == null) return null;
        if (x < 0) return null;
        if (y < 0) return null;
        if (x >= mapWidth) return null;
        if (y >= mapHeight) return null;
        return map.get(x).get(y);
	}

	@Override
	public List<MinesweeperPlayingPlayer> getPlayingPlayers() {
		return new ArrayList<MinesweeperPlayingPlayer>(this.playingPlayers);
	}

	@Override
	public List<MinesweeperPlayingPlayer> getRemainingPlayers() {
		List<MinesweeperPlayingPlayer> list = new ArrayList<MinesweeperPlayingPlayer>();
		for (MinesweeperPlayingPlayer pp : this.getPlayingPlayers()) {
			if (!pp.isEliminated()) list.add(pp);
		}
		return list;
	}

	@Override
	public int getMinesCount() {
		return this.minesCount ;
	}

	@Override
	public int getMinesLeft() {
		// TODO: Add null player to keep track of all the clicked mines by player null?
		int left = this.getMinesCount();
		for (MinesweeperPlayingPlayer player : this.playingPlayers)
			left -= player.getScore();
		return left;
	}

	public List<MinesweeperField> getAllUnclickedFields() {
		List<MinesweeperField> result = new ArrayList<MinesweeperField>();
		for (MinesweeperField mf : this) {
			if (!mf.isClicked()) result.add(mf);
		}
		return result;
	}

	public MinesweeperPlayingPlayer getPlayingPlayer(MinesweeperPlayer player) {
		for (MinesweeperPlayingPlayer pp : this.getPlayingPlayers()) {
			if (pp.getPlayer().equals(player)) return pp;
		}
		return null;
	}

	
	public MinesweeperPlayingPlayer getPlayingPlayer(String name) {
		for (MinesweeperPlayingPlayer player : this.playingPlayers) {
			if (player.getName().contentEquals(name)) return player;
		}
		return null;
	}
	
	public MinesweeperPlayingPlayer getCurrentPlayer() {
		if (this.playingPlayers.size() <= this.playerTurn) return null;
		return this.playingPlayers.get(this.playerTurn);
	}

	public int getPlayerTurn() {
		return this.playerTurn;
	}

	public int getPotentialWinnerCount() {
		int i = 0;
		this.eliminatePlayers();
		
		for (MinesweeperPlayingPlayer player : this.playingPlayers) {
			if (!player.isEliminated()) i++;
		}
		return i;
	}
	
	@Override
	public synchronized void nextTurn() {
		MinesweeperPlayingPlayer previousPlayer = this.getCurrentPlayer();
		
//		if (this.endCheckAndPerform()) { // This makes a winning bomb not be saved in replay.
//			return;
//		}
		
//		this.eliminatePlayers(); // This makes a winning bomb not be saved in replay.
		int oldTurn = this.playerTurn;
		do {
			this.playerTurn++;
			this.playerTurn = this.playerTurn % this.playingPlayers.size();
		}
		while (this.getCurrentPlayer().isEliminated() && oldTurn != this.playerTurn);
//		logger.error("next turn " + this, new AssertionError("called over and over from " + Thread.currentThread().getName()));
		
		executeEvent(new GameTurnChangeEvent(this, previousPlayer));
	}
	
	private boolean gameIsEnding = false;

	public boolean endCheckAndPerform() {
		if (!this.endCheck()) return false;
		if (gameIsEnding) return true;
		this.gameIsEnding = true;
		
		List<MinesweeperPlayingPlayer> allPlayers = new ArrayList<MinesweeperPlayingPlayer>(this.playingPlayers);
		for (final MinesweeperPlayingPlayer pp : allPlayers) {
			if (!pp.isEliminated()) {
				pp.eliminateWin();
			}
		}
		executeEvent(new GameEndedEvent(this));
		
		return true;
	}

	protected boolean endCheck() {
		return (this.getPotentialWinnerCount() <= endGameWhenPlayersLeftCount || this.getMinesLeft() == 0);
	}

	@Override
	public boolean isActionMode() {
		return this.actionMode;
	}

	@Override
	public void setActionMode(boolean actionMode) {
		this.actionMode = actionMode;
	}
	
	@Override
	public void clear() {
		this.moveHistory.clear();
		
		for (MinesweeperPlayingPlayer player : this.getPlayingPlayers()) {
			player.changeScore(-player.getScore()); // Basically set to zero
		}
		
		for (MinesweeperField field : this) {
			field.setMine(false);
			field.setValue(0);
			field.inactivate();
		}
	}
	
	private void eliminatePlayers() {
		int currentPointCap = Integer.MIN_VALUE;
		for (MinesweeperPlayingPlayer player : this.getPlayingPlayers()) {
			if (currentPointCap < player.getScore()) currentPointCap = player.getScore();
		}
		for (MinesweeperPlayingPlayer pp : this.getPlayingPlayers()) {
			if (!pp.isEliminated()) {
				if (executeEvent(new PlayerEliminateCheckEvent(pp.getMap(), pp)).isCancelled()) continue;
				
				if (pp.isEliminated()) continue; // Eliminated by plugin
				
				if (pp.hasMostScore()) continue; // TODO: Test disabling pp.hasMostScore() check. Considering the below line that checks if the player can reach more than point cap.
				if (this.getMinesLeft() + pp.getScore() >= currentPointCap) { // Can still make a tie
					continue;
				}
				
				pp.eliminateLoss();
			}
		}
	}
	
	void addSpecialMove(MinesweeperPlayingPlayer player, String moveInfo) {
		this.moveHistory.add(new SpecialMove(player, moveInfo));
	}
	
	@Override
	public void setCurrentPlayer(MinesweeperPlayingPlayer player) {
		this.setCurrentPlayerTurn(player.getIndex());
	}

	@Override
	public void setCurrentPlayerTurn(int playerIndex) {
		if (playerIndex >= this.playingPlayers.size()) throw new IllegalArgumentException("Player index " + playerIndex + " is out of range. Size " + this.playingPlayers.size());
		
		MinesweeperPlayingPlayer lastPlayer = this.getCurrentPlayer();
		this.playerTurn = playerIndex;
		
		executeEvent(new GameTurnChangeEvent(this, lastPlayer));
	}

	public List<MinesweeperField> getAllFields() {
		List<MinesweeperField> fields = new ArrayList<MinesweeperField>();
		for (MinesweeperField ff : this) {
			fields.add(ff);
		}
		return fields;
	}

	@Override
	public boolean isDraw() {
		int resultPos1 = 0;
		for (MinesweeperPlayingPlayer pp : this.getPlayingPlayers()) {
			if (pp.isEliminated() && pp.getResultPosition() == 1) resultPos1++;
		}
		return resultPos1 == 2;
//		return this.clickPos.contains("#DG");
	}
	
	@Override
	public boolean isGameOver() {
//		return this.endGameCalled || this.players.isEmpty();
		if (this.getPlayingPlayers().isEmpty()) return false; // game most likely not started yet.
		return this.getPlayersLeftCount() <= endGameWhenPlayersLeftCount;
	}
	
	@Override
	public int getPlayersLeftCount() {
		int i = 0;
		for (MinesweeperPlayingPlayer pp : this.playingPlayers) {
			if (!pp.isEliminated()) i++;
		}
		return i;
	}

    @Override
    public Iterator<MinesweeperField> iterator() {
        return new MapListIterator<MinesweeperField>(this.map);
    }

    @Override
	public Iterable<MinesweeperField> getIteration() {
        return this;
	}

	@Override
	public String saveMap() {
		String str = "";
		for (int yy = 0; yy < this.getFieldHeight(); yy++) {
			String st = "";
			for (int xx = 0; xx < this.getFieldWidth(); xx++) {
				MinesweeperField field = this.getPosition(xx, yy);
				String fieldString = "";
				if (!field.isClicked()) fieldString = field.isMine() ? "x" : "_";
				else if (field.isBlocked()) fieldString = ":";
				else if (field.isMine()) {
					if (field.getWhoClicked() == null) fieldString = "*";
					else {
						fieldString = "" + (char)('a' + field.getWhoClicked().getIndex());
					}
				}
				else {
					fieldString = Integer.toString(field.getValue(), 16);
				}
				st += fieldString;
			}
			str += st;
			if (yy < this.getFieldHeight() - 1) str += "-";
		}
		return str;
	}
	@Override
	public String[] saveMapMultiline() {
		return this.saveMap().split("-");
	}

	@Override
	public MinesweeperMap loadMap(String data) {
		this.clear();
		int xsize = 16;
		boolean doInitFields = false;
		int xx = 0;
		int yy = 0;
		String str2 = data.replace(';', '&');
		
		int strlen = str2.length();
//		logger.info("Load Map String: " + str2);
		for (int i = 0; i < strlen; i++) {
			char c = str2.charAt(i);
//			trace(i, yy, xx, c, 'Code', str2.charCodeAt(i));
			if (c == 9) continue;// skip tabs
			if (c == '&') break;
			if (c == '[') break;
			if (c == ' ') continue;
			if ((c == '\r') || (c == '-')) {
				if (xx >= 1) yy++;
				xx = 0;
				continue;
			}
			if ((xsize > 0) && (xx >= xsize)) {
//				trace("xx >= xsize, increase yy");
				yy++;
				xx = 0;
			}
			if (c == '\n') {
				if (xx >= 1) yy++;
				xx = 0;
				continue;
			}
			MinesweeperField f = this.getPosition(xx, yy);
//			logger.info("f for " + xx + ", " + yy + " = " + f);
			if ((c == '_') || (c == 'w') || (c == '?')) f.inactivate();
			else if (c == ':') f.setBlocked(true);
			else {
				if (f == null) {
					throw new NullPointerException("f for " + xx + ", " + yy + " is null");
				}
				if (c == 'x') {
					f.setMine(true);
					doInitFields = true;
				}
				else if ((c >= 'a' && c <= 'h') || (c == '*')) { // a..h = 8 players, * == unset player.
					f.setMine(true);
					if (c == '*') f.activate((MinesweeperPlayingPlayer)null);
					else f.activate(this.getPlayingPlayers().get(c - 'a'));
				}
				
				if (f.isMine()) {
				}
				else if (c >= '0' && c <= '9') f.setValue(c - '0');
				
				if (c != 'x' && !f.isClicked()) f.activate();
			}
			xx++;
		}
		
//		int mleft = 51 - minesfound;
		if (doInitFields) this.initFields();
		
		this.sendUpdateToPlayers();
		return this;
	}

	@Override
	@Deprecated
	public void copyFrom(MinesweeperMap game) {
		// TODO: For GameReplay API: Game copy - copy players - except names, use the names we have. If you need more players, use temp-names.
		
		for (MinesweeperField mf : this) {
			mf.copyFrom(game.getPosition(mf.getX(), mf.getY()));
		}
	}

	public MinesweeperField getPosition(String xy) {
		int x = Integer.parseInt(xy.substring(0, xy.length() / 2), POSITION_RADIX);
		int y = Integer.parseInt(xy.substring(xy.length() / 2), POSITION_RADIX);
		return this.getPosition(x, y);
	}

	@Override
	public void setMinePositions(String minepos, int charsPerPos) {
		this.clear();
		if (minepos == null) {
			this.generate();
			return;
		}
			
		if (minepos.startsWith("*")) throw new IllegalArgumentException("Decrypt the string before passing it to setMinePositions.");
		
		while (!minepos.isEmpty()) {
			MinesweeperField pos = this.getPosition(minepos.substring(0, charsPerPos));
			pos.setMine(true);
			minepos = minepos.substring(charsPerPos);
		}
		this.postGenerate(); // init fields called here also.
	}

	
	@Override
	public final void sendUpdateToPlayers() {
		executeEvent(new GameSendUpdateEvent(this));
		// TODO: Replace sendUpdateToPlayers with MinesweeperEvent
	}

	@Override
	public void setMinePositions(String minepos) {
		int charsPerPos = 2;
		
		if (this.getFieldWidth()  > 16) throw new IllegalStateException("Error: Map Width  is more than 16 and charsPerPos is not defined (using default 2).");
		if (this.getFieldHeight() > 16) throw new IllegalStateException("Error: Map Height is more than 16 and charsPerPos is not defined (using default 2).");
		
		this.setMinePositions(minepos, charsPerPos);
	}

	public MinesweeperField newTile(MfeMap map, int x, int y) {
        return new MfeField(this, x, y);
    }
	
	public MoveAllowedState performClicks(String clicks) {
		while (clicks.length() > 0) {
			MinesweeperMove move;
			try {
				String moveString = clicks.substring(0, GameReplay.CHARS_PER_CLICK);
				if (moveString.startsWith("#")) this.handleSpecialMove(moveString);
				
//				move = MinesweeperMove.createFromString(this, moveString);
				move = this.getCurrentPlayer().createMove(moveString);
			}
			catch (IllegalArgumentException e) {
				throw new IllegalStateException("Unable to perform clicks for game " + this, e);
			}
			clicks = clicks.substring(GameReplay.CHARS_PER_CLICK);
			MoveAllowedState moveResult = performMove(move); // Move is added to movehistory here
			if (!moveResult.isOK()) return moveResult;
		}
		return MoveAllowedState.OK;
	}
	
	
	private void handleSpecialMove(String moveString) {
		int playerIndex;
		try {
			playerIndex = Integer.parseInt(UtilZomisUtils.substr(moveString, -1)) - 1; // #D1 == player index 0 draw proposal
		}
		catch (NumberFormatException e) {
			playerIndex = -1;
		}
		MfePlayer pp;
		if (moveString.startsWith("#S")) {
			if (playerIndex == -1)
				pp = (MfePlayer) this.getCurrentPlayer();
			else pp = (MfePlayer) this.getPlayingPlayers().get(playerIndex);
			pp.surrender();
		}
		if (moveString.startsWith("#W")) {
			if (playerIndex == -1)
				pp = (MfePlayer) this.getCurrentPlayer();
			else pp = (MfePlayer) this.getPlayingPlayers().get(playerIndex);
			pp.walkover();
		}
		if (moveString.startsWith("#D")) {
			if (playerIndex == -1) {
				pp = null;
				for (MinesweeperPlayingPlayer pp2 : this.getPlayingPlayers()) {
					if (!((MfePlayer) pp2).hasDraw()) {
						// TODO: DRAW the entire game, that is: Draw proposal for all who have not drawed already but here only one player is set to draw!?
						pp = (MfePlayer) this.getCurrentPlayer();
					}
				}
			}
			else pp = (MfePlayer) this.getPlayingPlayers().get(playerIndex);
			
			if (pp != null) pp.proposeDraw();
		}
	}
	@Override
	public MinesweeperMove createMove(MinesweeperPlayingPlayer player, MinesweeperWeapon weapon, MinesweeperField field) {
		return new MfeMove(player, player == null ? weapon : player.hasWeapon(weapon), field);
	}
	@Override
	public final boolean hasPlugin(MinesweeperPlugin plugin) {
		return this.activePlugins.contains(plugin);
	}
	@Override
	public final void addPlugin(MinesweeperPlugin plugin) {
		this.activePlugins.add(plugin);
	}
	
	@Override
	public final List<MinesweeperPlugin> getActivePlugins() {
		return new ArrayList<MinesweeperPlugin>(this.activePlugins);
	}

	@Override
	public MapFactory getMapFactory() {
		return MapFactoryImpl.newFactory();
	}
	/**
	 * Attempts to make the move
	 * @return The result of the move
	 */
	@Override
	public MoveAllowedState performMove(MinesweeperMove move) {
		if (move instanceof SpecialMove) { // TODO: Code-crap! Don't use instanceof like this
			return ((SpecialMove)move).performSpecialMove();
		}
		
		// TODO: Cleanup MinesweeperMove.performMove method.
		
		CancellableEvent event = executeEvent(new PlayerMoveEvent(move));
		if (event.isCancelled()) {
			return MoveAllowedState.DENIED_BY_PLUGIN;
		}

		MinesweeperWeapon weapon = move.getPlayer().hasWeapon(move.getWeapon());
		if (weapon == null) {
			return MoveAllowedState.WEAPON_NOT_FOUND; // throw new NullPointerException("Player " + player.toString() + " does not have weapon " + this.weapon.toString());
		}

		if (!move.getMoveAllowedState().isOK()) {
			return move.getMoveAllowedState();
		}
		
		if (!weapon.canUseAt(move.getPlayer(), move.getMovePosition())) {
			return move.getMoveAllowedState();
		}
		
		boolean useOK = move.getWeapon().useAt(move);
		if (!useOK && (move.getPlayer().getPlayer() != null)) {
			return MoveAllowedState.MOVE_FAILED;
		}
		if (!useOK)
			return MoveAllowedState.MOVE_FAILED;
		
		this.postMove(move);
		return MoveAllowedState.OK;
	}
	@Override
	public MinesweeperField getPosition(MinesweeperField field) {
		return this.getPosition(field.getX(), field.getY());
	}
	@Override
	public MinesweeperReplay createReplay() {
		MinesweeperMap mmap = MapUtils.copyBase(this);
		
		return new GameReplay(MapUtils.getMinesString(this), MapUtils.getClickString(this), mmap);
	}

    @Override
    public <T extends BaseEvent> T executeEvent(T event) {
        return this.events.executeEvent(event);
    }

}
