package net.zomis.minesweeper.game.model;

import java.util.ArrayList;
import java.util.List;

import net.zomis.UtilZomisList;
import net.zomis.UtilZomisList.FilterInterface;
import net.zomis.minesweeper.api.ai.MinesweeperAI;
import net.zomis.minesweeper.api.MinesweeperPlayer;
import net.zomis.minesweeper.events.game.GameScoreChanged;
import net.zomis.minesweeper.events.game.PlayerEliminatedEvent;
import net.zomis.minesweeper.game.MinesweeperField;
import net.zomis.minesweeper.game.MinesweeperGame;
import net.zomis.minesweeper.game.MinesweeperMap;
import net.zomis.minesweeper.game.MinesweeperMove;
import net.zomis.minesweeper.game.MinesweeperPlayingPlayer;
import net.zomis.minesweeper.weapons.MinesweeperWeapon;

/**
 * A class representing a {@link MinesweeperPlayer} that is playing in a {@link MinesweeperGame}
 */
public class MfePlayer implements MinesweeperPlayingPlayer {
	private final List<MinesweeperWeapon> weapons;
	private final MinesweeperMap map;
	private final int index;
	
	private MinesweeperPlayer player; // changed in "updatePlayer" (when a player logins again)
	private int score;
	private boolean gameOver;
	private int resultPosition;
	private boolean didSurrender;
	private boolean didWalkover;

	public MfePlayer(MinesweeperMap map, MinesweeperPlayer player, int index) {
//		this.map = map == null ? null : map.getMapFactory().hiddenMap(map);
		this.map = map;
		this.player = player;
		
		this.weapons = new ArrayList<MinesweeperWeapon>();
		this.index = index;
		
		this.reset();
	}
	
	@Override
	public void reset() {
		for (MinesweeperWeapon weapons : this.weapons)
			weapons.resetUsage();
		
		this.score = 0;
		this.gameOver = false;
		this.resultPosition = -1;
		this.didSurrender = false;
		this.didWalkover = false;
	}

	@Override
	public MinesweeperPlayer getPlayer() {
		return this.player;
	}
	
	@Override
	public String getName() {
		if (this.player == null) return "Player " + this.getIndex();
		
		if (this.isAI()) return this.player.getName().substring(1);
		
		return this.player.getName();
	}

	@Override
	public MinesweeperMap getMap() {
		return this.map;
	}

	@Override
	public void giveWeapon(MinesweeperWeapon weapon) {
		this.weapons.add(weapon);
	}

	@Override
	public boolean canUseWeapon(String weaponType) {
		MinesweeperWeapon weapon = this.getWeapon(weaponType);
		if (weapon == null) return false;
		return weapon.canUse(this);
	}
	
	@Override
	public int getScore() {
		return this.score;
	}

	@Override
	public void changeScore(int change) {
		// TODO: Check if there has been a radical change now when a GameScoreChanged event is executed within MfePlayer (for each score change instead of only field activate)
		if (change == 0) return; // This is not what I call a "change" exactly...
		int old = this.getScore();
		this.score += change;
		MinesweeperEvents.executeEvent(new GameScoreChanged(this.getMap(), this, old));
	}

	@Override
	public MinesweeperWeapon hasWeapon(MinesweeperWeapon weapon) {
		for (MinesweeperWeapon weap : this.weapons)
		if (weap.equals(weapon))
			return weap;
		return null;
	}

	@Override
	public boolean isMyTurn() {
		return this.getIndex() == this.getMap().getPlayerTurn() || this.getMap().isActionMode();
	}

	@Override
	public boolean hasMostScore() {
		for (MinesweeperPlayingPlayer player : this.getMap().getPlayingPlayers()) {
			if (player.getScore() > this.getScore()) return false;
		}
		return true;
	}
	
	@Override
	public boolean isEliminated() {
		return this.gameOver;
	}
	
	@Override
	public int getIndex() {
		return this.index;
	}
	
	@Override
	public void eliminateLoss() {
		this.eliminate(true);
	}
	@Override
	public void eliminateWin() {
		this.eliminate(false);
	}
	
	private void eliminate(boolean lost) {
//		logger.info("Player " + this + " eliminated! Unknown result position. Lost? " + lost);
		
		int playerResultPosition = this.getMap().getPlayingPlayers().size() + 1; // if no one else has been eliminated, the player is at 1st place. Because the player itself has not been eliminated, it should get increased below.
		if (!lost) playerResultPosition = 0;
		
		boolean posTaken = false;
		do {
			posTaken = false;
//			logger.info("Eliminate " + this + " resultpos loop is " + playerResultPosition);
			playerResultPosition += (lost ? -1 : +1);
			for (MinesweeperPlayingPlayer pp : this.getMap().getPlayingPlayers()) {
				if (pp.isEliminated() && pp.getResultPosition() == playerResultPosition) {
//					logger.info("Eliminate " + this + " player match " + pp + " is " + playerResultPosition);
					posTaken = true;
					break;
				}
			}
		}
		while (posTaken && playerResultPosition >= -25 && playerResultPosition < 25);
		
		this.eliminate(playerResultPosition);
	}
	
	@Override
	public void eliminate(int resultPosition) {
		// AI Eliminate is handled by a MfeAI class, which calls this method as a super method.
		if (this.isEliminated()) return; // Can't be eliminated more than once.
		
		this.gameOver = true;
		this.resultPosition = resultPosition;
		
		MinesweeperEvents.executeEvent(new PlayerEliminatedEvent(this));
		
		// Prevent game from being stuck because a player who's turn it was is eliminated.
		if (this.getMap().getPlayerTurn() == this.getIndex()) {
			this.getMap().nextTurn();
		}
	}
	
	@Override
	public String toString() {
		return this.getIndex() + ": " + this.player + " (" + this.getScore() + ")";
	}

	@Override
	public boolean isAI() {
		return false;
	}

	@Override
	public int getResultPosition() {
		return this.resultPosition;
	}

	@Override
	public List<MinesweeperWeapon> getWeapons() {
		return new ArrayList<MinesweeperWeapon>(this.weapons);
	}
	
	@Override
	public List<MinesweeperMove> getMoves() {
		List<MinesweeperMove> myMoves = this.getMap().getMoveHistory();
		
		UtilZomisList.filter(myMoves, new MyMovesFilter(this));
		return myMoves;
	}

	private static class MyMovesFilter implements FilterInterface<MinesweeperMove> {
		private MfePlayer pp;
		
		MyMovesFilter(MfePlayer pp) {
			this.pp = pp;
		}

		@Override
		public boolean shouldKeep(MinesweeperMove obj) {
			return obj.getPlayer().equals(pp);
		}
	}
	
	public void updatePlayer(MinesweeperPlayer old, MinesweeperPlayer real) {
		if (this.player == null) return;
		if (this.player.equals(old)) this.player = real;
	}

	@Override
	public MinesweeperWeapon getWeapon(String weaponType) {
		for (MinesweeperWeapon weapon : this.weapons)
		if (weapon.getWeaponType().contentEquals(weaponType))
			return weapon;
		return null;
	}

	
	
	

	
	
	/*
	 * SURRENDER, DRAW, WALKOVER implementation below.
	 *
	 */
	
	public void surrender() {
		this.didSurrender = true;
		((MfeMap)this.getMap()).addSpecialMove(this, "#S" + (this.getIndex()+1));
		this.eliminateLoss();
	}

	public void walkover() {
		this.didWalkover = true;
		((MfeMap)this.getMap()).addSpecialMove(this, "#W" + (this.getIndex()+1));
		this.eliminateLoss();
	}
	
	public boolean proposeDraw() {
		if (!MfePlayer.drawPossible(this.map)) return false;
		
		if (this.hasDraw()) return true;
		
		boolean drawAgreement = true;
		
		// Add a 'draw proposal' mark.
		((MfeMap)this.getMap()).addSpecialMove(this, "#D" + (this.getIndex()+1));
		
		// Make it possible for two AIs to draw against each other
		if (this.getMap().getPlayersLeftCount() == 2) {
			for (MinesweeperPlayingPlayer pp : this.getMap().getPlayingPlayers())
			if (pp.getIndex() != this.getIndex()) {
				if (pp.isAI() && !((MfePlayer)pp).hasDraw()) {
					if (pp.getAI().agreeDraw(pp)) {
                        pp.proposeDraw();
                    }
				}
			}
		}
		
		// Check if everyone has agreed to draw
		for (MinesweeperPlayingPlayer pp : this.getMap().getPlayingPlayers())
		if (pp != this) {
			if (!pp.isEliminated()) {
				if (!((MfePlayer)pp).hasDraw()) {
					drawAgreement = false;
					break;
				}
			}
		}
		
		if (drawAgreement) {
			// End the game in a draw.
			((MfeMap)this.getMap()).addSpecialMove(this, "#DW");
			for (MinesweeperPlayingPlayer pp : this.getMap().getPlayingPlayers()) {
				if (!pp.isEliminated()) {
					pp.eliminate(1);
				}
			}
		}
		return true;
	}
	
	public boolean hasSurrender() {
		return this.didSurrender;
	}

	public boolean hasWalkover() {
		return this.didWalkover;
	}

	public static boolean drawPossible(MinesweeperMap game) {
		return game.getMinesLeft() <= game.getMinesCount() - 20;
	}
	
	public boolean hasDraw() {
//		return this.drawed;
		
		List<MinesweeperMove> history = this.getMap().getMoveHistory();
		for (int i = history.size() - 1; i >= 0; i--) {
			if (history.get(i) instanceof SpecialMove) {
				SpecialMove spec = (SpecialMove) history.get(i);
				if (!spec.getType().equals("D")) return false;
				else if (spec.getPlayerIndex() == this.getIndex()) {
					return true;
				}
			}
			else {
				return false;
			}
		}
		return false;
	}

	@Override
	public MinesweeperAI getAI() {
		return null;
	}

	@Override
	public MinesweeperMove createMove(String weaponType, MinesweeperField field) {
		return this.createMove(this.getWeapon(weaponType), field);
	}

	@Override
	public MinesweeperMove createMove(MinesweeperWeapon weapon, MinesweeperField field) {
		return new MfeMove(this, this.hasWeapon(weapon), field);
	}
	@Override
	public MinesweeperMove createMove(String moveData) {
//		if (string.length() < 3) throw new IllegalArgumentException("Expected at least 3 chars in string to create Minesweeper move. Found: " + string);
//		if (string.length() % 2 == 0) throw new IllegalArgumentException("Expected an odd amount of characters in string, at least 3 chars. Found: " + string);
		if (moveData.length() != GameReplay.CHARS_PER_CLICK) throw new IllegalArgumentException("Move data must have length " + GameReplay.CHARS_PER_CLICK + " data was " + moveData);
		
		if (moveData.startsWith("#")) return new SpecialMove(this, moveData);
		else {
			if (moveData.contentEquals("W##")) return new SpecialMove(this, "#W" + (this.getIndex() + 1));
			
			String xy = moveData.substring(1);
			
			String x = xy.substring(0, xy.length() / 2);
			String y = xy.substring(xy.length() / 2);
			try {
				return new MfeMove(this, this.getWeapon(moveData.substring(0, 1)), this.getMap().getPosition(Integer.parseInt(x, 16), Integer.parseInt(y, 16)));
			}
			catch (NumberFormatException e) {
				throw new IllegalArgumentException("Position-part of string is not parsable. Expected weapontype + Hexadecimal position (x and y). Found: " + moveData);
			}
		}
	}
}
