package net.zomis.minesweeper.game.model;

import java.io.File;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import net.zomis.minesweeper.api.Minesweeper;
import net.zomis.minesweeper.api.MinesweeperPlayer;
import net.zomis.minesweeper.api.MinesweeperPlugin;
import net.zomis.minesweeper.api.PluginHelper;
import net.zomis.minesweeper.events.BaseEvent;
import net.zomis.minesweeper.events.Command;
import net.zomis.minesweeper.events.Event;
import net.zomis.minesweeper.events.EventListener;
import net.zomis.minesweeper.events.game.GameEvent;
import net.zomis.minesweeper.events.player.PlayerCommandEvent;
// import net.zomis.minesweeper.events.plugins.PluginEnableEvent;

public class MinesweeperEvents {
	// TODO: Extend EventExecutor from net.zomis.events
	private Map<Class<? extends BaseEvent>, Collection<EventHandler>> bindings;
	private Map<String, EventHandler> commands;
	
	private static MinesweeperEvents eventHandler;

	public static Set<String> getRegistredCommands() {
		return new HashSet<String>(getCreate().commands.keySet());
	}
	public static EventHandler getCommand(String command) {
		return eventHandler.commands.get(command);
	}
	
	public MinesweeperEvents activatePlugin(MinesweeperPlugin plugin, PluginHelper helper, File file) {
		if (plugin.isEnabled()) 
			throw new IllegalArgumentException("Plugin " + plugin + " is already enabled");

        plugin.initialize(Minesweeper.getServer(), helper, file);
		plugin.setEnabled(true);
        // MinesweeperEvents.executeEvent(new PluginEnableEvent(plugin));
		return this;
	}
	
	private MinesweeperEvents() {
		this.bindings = new HashMap<Class<? extends BaseEvent>, Collection<EventHandler>>();
		this.commands = new HashMap<String, EventHandler>();
		if (eventHandler == null) eventHandler = this;
		else throw new AssertionError("An instance of " + this.getClass().getSimpleName() + " already exists.");
	}
	
	@Deprecated
	public static int getListenerCountFor(Class<? extends BaseEvent> clazz) {
		if (!eventHandler.bindings.containsKey(clazz)) return 0;
		return eventHandler.bindings.get(clazz).size();
	}

	public static boolean canHandleCommand(String command, MinesweeperPlayer user) {
		return eventHandler.commands.containsKey(command);
		// Whether or not user has access to the command, is handled in MfeUser.hasAccessToCommand
	}
	
	/**
	 * Alerts all plugins about an event.
	 * 
	 * @param event The event
	 * @return The same event
	 */
	public static <T extends BaseEvent> T executeEvent(T event) {
//		logger.info("Execute Event: " + event.getClass().getSimpleName());
		if (getCreate() == null)
			return event;
		if (eventHandler.bindings == null) 
			return event;
		
		Collection<EventHandler> handlers = null;
		
		if (!(event instanceof PlayerCommandEvent)) {
			if (!eventHandler.bindings.containsKey(event.getClass())) return event;
			handlers = eventHandler.bindings.get(event.getClass());
		}
		else {
			String command = ((PlayerCommandEvent)event).getCommand();
			EventHandler eh = eventHandler.commands.get(command);
			if (eh == null) {
				return event;
			}
			try {
				eh.execute(event);
			} catch (Exception e) {
                // logger.error(String.format("Error performing command %s in %s: %s", event.toString(), eh.toString(), e.getMessage()));
                throw new RuntimeException(e);
			}
			return event;
		}
		
//		logger.info("Events has " + handlers.size() + " handlers.");
		for (EventHandler handler : handlers) {
			if (handler == null) 
				continue; // should not happen, but you never know...
			
			if (event instanceof GameEvent) { // Check if game has plugin activated before actually calling the handler
				GameEvent g = (GameEvent) event;

				if (!g.getMap().hasPlugin(handler.getPlugin())) {
//					logger.debug(g.getMap() + " map does not have plugin " + handler.getPlugin());
					continue;
				}
			}
			
//			logger.info("Execute Event handler: " + handler.toString()); // info because it is quite good to know which event handlers are being called.
			try {
				handler.execute(event);
			} catch (Exception e) {
                throw new RuntimeException(e);
				// logger.error(String.format("Error handling event %s in %s: %s", event.toString(), handler.toString(), e.getMessage()), e);
			}
		}
		return event;
	}
	
	public static MinesweeperEvents getCreate() {
		if (MinesweeperEvents.eventHandler == null) MinesweeperEvents.eventHandler = new MinesweeperEvents();
		return MinesweeperEvents.eventHandler;
	}
	
	
	public void registerListener(final EventListener listener, final MinesweeperPlugin plugin) {
		if (this.bindings == null)
			throw new NullPointerException("No bindings on MinesweeperEvents has been created.");
		// Do NOT check here if listener is already registered since listeners can be added by many different plugins.
		Method[] methods = listener.getClass().getDeclaredMethods();

		for (final Method method : methods) {
			Event annotation = method.getAnnotation(Event.class);
			Command commandAnnotation = method.getAnnotation(Command.class);
			if (annotation == null && commandAnnotation == null) {
//				logger.debug("Method does not have Event or Command annotation: " + method.getName());
				continue;
			}
			
			Class<?>[] parameters = method.getParameterTypes();
			if (parameters.length != 1) continue;
			
			Class<?> param = parameters[0];
			
			if (!method.getReturnType().equals(void.class)) {
                throw new IllegalArgumentException("Method has non-void return: " + method.getName());
			}
			
			if (PlayerCommandEvent.class.isAssignableFrom(param)) {
				if (commandAnnotation == null) {
                    throw new IllegalArgumentException("PlayerCommandEvent should use the @Command annotation instead of the @Event annotation.");
				}
				
				if (this.commands.containsKey(commandAnnotation.command())) {
                    throw new IllegalStateException("Command " + commandAnnotation.command() +
                        " was registred by " + this.commands.get(commandAnnotation.command()).getPlugin().getSimpleName()
                        + ", cannot register " + method.getName() + " in " + plugin.getSimpleName());
				}
				EventHandler eh = new EventHandler(plugin, listener, method, commandAnnotation);
				this.commands.put(commandAnnotation.command(), eh);
			}
			else if (BaseEvent.class.isAssignableFrom(param)) {
				@SuppressWarnings("unchecked")
				Class<? extends BaseEvent> realParam = (Class<? extends BaseEvent>) param;

				// Get the collection of all events of this class
				if (!this.bindings.containsKey(realParam)) {
					this.bindings.put(realParam, new HashSet<EventHandler>());
				}
				Collection<EventHandler> set = this.bindings.get(realParam);

				// Create a new event handler and add it to the collection
				EventHandler eh = new EventHandler(plugin, listener, method, annotation);
				set.add(eh);
			}
		}
	}
	/**
	 * Removes a listener from bindings
	 * @param listener The {@link EventListener} to remove
	 */
	public static void removeListenerFromBindings(EventListener listener) {
		for (Entry<Class<? extends BaseEvent>, Collection<EventHandler>> ee : eventHandler.bindings.entrySet()) {
			Iterator<EventHandler> it = ee.getValue().iterator();
			while (it.hasNext()) {
				EventHandler eh = it.next();
				if (eh.getListener() == listener) {
					it.remove();
				}
			}
		}
	}
	public static void clearListeners() {
		eventHandler.bindings.clear();
		eventHandler.commands.clear();
	}
	@Deprecated 		// Unused method
	public static boolean pluginListensForEvent(MinesweeperPlugin plugin, Class<?> eventClass) {
		if (!eventHandler.bindings.containsKey(eventClass)) return false;
		
		Collection<EventHandler> coll = eventHandler.bindings.get(eventClass);
		for (EventHandler eh : coll) {
			if (eh.getPlugin().equals(plugin)) return true;
		}
		return false;
	}
}
