package net.zomis.minesweeper.game.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;

import net.zomis.custommap.model.GenericTileModel;
import net.zomis.minesweeper.api.FieldType;
import net.zomis.minesweeper.events.game.FieldActivatedEvent;
import net.zomis.minesweeper.events.game.FieldUpdateEvent;
import net.zomis.minesweeper.game.MinesweeperField;
import net.zomis.minesweeper.game.MinesweeperMap;
import net.zomis.minesweeper.game.MinesweeperMove;
import net.zomis.minesweeper.game.MinesweeperPlayingPlayer;
import net.zomis.minesweeper.weapons.MinesweeperWeapon;

public abstract class MfeField extends GenericTileModel<MinesweeperField, MfeMap> implements MinesweeperField {
	private int mineValue = 0;
	private int neighboringMines = 0;
	private MinesweeperPlayingPlayer clickedBy = null;
	private boolean clicked = false;
	private final Collection<MinesweeperField> neighbors;
	private final Collection<MinesweeperField> invertedNeighbors;
	private boolean blocked;
	
	public MfeField(MfeMap map, int x, int y) {
		super(map, x, y);
		this.neighbors = new HashSet<MinesweeperField>();
		this.invertedNeighbors = new HashSet<MinesweeperField>();
	}
	
	@Override
	public boolean isClicked() {
		return this.clicked || this.isBlocked();
	}
	@Override
	public MinesweeperPlayingPlayer getWhoClicked() {
		return this.clickedBy;
	}
	@Override
	public boolean isDiscoveredMine() {
		return this.isClicked() && this.mineValue != 0;
	}
	@Override
	public int getKnownValue() {
		if (!this.isClicked() || this.isMine()) return Integer.MIN_VALUE;
		return this.neighboringMines;
	}
	
	private void activate(MinesweeperPlayingPlayer player, MinesweeperMove moveInfo) {
		if (this.isMine() && this.clickedBy != null) {
			// Reduce score for old player who clicked
			if (this.map instanceof MfeMap)
				((MfeMap) this.map).performScoreChange(this.clickedBy, -this.mineValue);
		}

		this.clicked = true;
		this.clickedBy = player;
		
		MinesweeperEvents.executeEvent(new FieldActivatedEvent(this, moveInfo));

		if (this.isMine() && this.clickedBy != null) {
			// Increase score for player who clicked
			if (this.getMap() instanceof MfeMap)
				((MfeMap) this.getMap()).performScoreChange(this.clickedBy, this.mineValue);
		}
		
	}
	@Override
	public void activate(MinesweeperPlayingPlayer player) {
		this.activate(player, player == null ? null : player.createMove((MinesweeperWeapon) null, this));
	}
	
	@Override
	public void activate(MinesweeperMove moveInfo) {
		this.activate(moveInfo == null ? null : moveInfo.getPlayer(), moveInfo);
	}
	@Override
	public void activate() {
		this.activate(null, null);
	}
	@Override
	public void inactivate() {
		this.clicked = false;
		if (this.clickedBy != null && this.isMine()) this.clickedBy.changeScore(-this.getMineValue());
		this.clickedBy = null; // perform this too? Not really needed since there is a `clicked` boolean.
	}
	@Override
	public Collection<MinesweeperField> getNeighbors() {
		return new ArrayList<MinesweeperField>(this.neighbors); // arraylist instead of HashSet to support 2 equal neighbors.
	}
	@Override
	public void addNeighbor(MinesweeperField field) {
		if (field != null && field != this) {
			this.neighbors.add(field);
			((MfeField)field).invertedNeighbors.add(this);
		}
	}
	@Override
	public boolean removeNeighbor(MinesweeperField field) {
		((MfeField)field).invertedNeighbors.remove(this);
		return this.neighbors.remove(field);
	}
	@Override
	public MinesweeperMap getMap() {
		return this.map;
	}
	@Override
	public void init() {
		// initialize value
		int value = 0;
		for (MinesweeperField field : this.neighbors) {
			value += field.getMineValue();
		}
		this.neighboringMines = value;
	}
	@Override
	public boolean isMine() {
		return this.getMineValue() != 0;
	}
	@Override
	public int getValue() {
		return this.neighboringMines;
	}

	@Override
	public int getMineValue() {
		return this.mineValue;
	}

	@Override
	public void setMine(boolean mine) {
		this.mineValue = (mine ? 1 : 0);
	}
	@Override
	public String getCoordinate() {
		int radix = 16;
		return Integer.toString(this.getX(), radix) + Integer.toString(this.getY(), radix);
	}

	@Override
	public void addNeighbor(int dx, int dy) {
		this.addNeighbor(this.getMap().getPosition(this.getX() + dx, this.getY() + dy));
	}
	
	@Override
	public String toString() {
		if (this.isClicked()) {
			return this.getCoordinate() + "=" + (this.isBlocked() ? "#" : this.isMine() ? "*" : this.getValue());
		}
		else {
			return this.getCoordinate();
		}
	}

	@Override
	public void setValue(int value) {
		this.neighboringMines = value;
	}

	@Override
	public Collection<MinesweeperField> getInvertedNeighbors() {
		return new ArrayList<MinesweeperField>(this.invertedNeighbors);
	}

	@Override
	public MinesweeperField getRelativePosition(int deltaX, int deltaY) {
		return (MinesweeperField) super.getRelative(deltaX, deltaY);
	}

	@Override
	public void copyFrom(MinesweeperField position) {
		this.clicked = position.isClicked();
		this.clickedBy = (position.getWhoClicked() == null ? null : this.getMap().getPlayingPlayers().get(position.getWhoClicked().getIndex())); // copy index, not reference.
		this.mineValue = position.getMineValue();
		this.setBlocked(position.isBlocked());
		this.neighboringMines = position.getValue();
		
		Iterator<MinesweeperField> it = this.getNeighbors().iterator();
		while (it.hasNext()) { this.removeNeighbor(it.next()); }
		
		for (MinesweeperField newNeighbor : position.getNeighbors()) {
			this.addNeighbor(this.getMap().getPosition(newNeighbor.getX(), newNeighbor.getY()));
		}
		
	}
	
	@Override
	public boolean isBlocked() {
		return this.blocked;
	}
	
	@Override
	public void setBlocked(boolean blocked) {
		this.blocked = blocked;
	}

	@Override
	@Deprecated
	public int getFieldType() {
		return (this.isBlocked() ? TYPE_BLOCKED : (this.isClicked() ? TYPE_CLICKED : TYPE_UNCLICKED));
	}

	@Override
	public void sendInfoToPlayers() {
		MinesweeperEvents.executeEvent(new FieldUpdateEvent(this));
	}

	@Override
	public void sendInfoToPlayer(MinesweeperPlayingPlayer player) {
		if (player.getPlayer() == null) return;
		player.getPlayer().sendFieldInfo(this, this.getWhoClicked(), false);
	}
	@Override
	public boolean equals(Object obj) {
		if (!(obj instanceof MinesweeperField)) return false;
		
		MinesweeperField ff = (MinesweeperField) obj;
		
		return this.getX() == ff.getX() && this.getY() == ff.getY();
	}

	@Override
	public FieldType getFieldStyle() {
		return (this.isBlocked() ? FieldType.BLOCKED : (this.isClicked() ? FieldType.CLICKED : FieldType.UNCLICKED));
	}
	
}
