package net.zomis.minesweeper.game;

import java.util.Collection;

import net.zomis.minesweeper.api.FieldType;

public interface MinesweeperField {
	@Deprecated
	static int TYPE_UNCLICKED = 0;
	@Deprecated
	static int TYPE_CLICKED = 1;
	@Deprecated
	static int TYPE_BLOCKED = 2;
	
	/**
	 * @return X-position of this field
	 */
	int getX();
	
	/**
	 * @return Y-position of this field
	 */
	int getY();
	
	/**
	 * Checks if this field has been clicked at. (It is revealed)
	 * @return True if the field has been clicked.
	 */
	boolean isClicked();
	
	/**
	 * Finds out who clicked on this field.
	 * 
	 * @return The player that clicked on this field, or null if it has not been clicked.
	 */
	MinesweeperPlayingPlayer getWhoClicked();

	/**
	 * Checks if this field contains a mine that has been found.
	 * 
	 * @return True if this mine has been discovered, false otherwise
	 */
	boolean isDiscoveredMine();
	
	/**
	 * Get the value of the field.
	 * 
	 * If the field is not displayed, this should return Integer.MIN_VALUE
	 * @return Integer.MIN_VALUE if not displayed, the corresponding value of the field otherwise. 
	 */
	int getKnownValue();
	
	/**
	 * The neighbors where this field is looking for mines to determine it's value.
	 * @return Collection of {@link MinesweeperField} neighbors
	 * @see #getInvertedNeighbors()
	 */
	Collection<MinesweeperField> getNeighbors();

	/**
	 * Gets all {@link MinesweeperField}s that has this field as neighbor 
	 * @return Collection of {@link MinesweeperField}s that has this field as neighbor.
	 * @see #getNeighbors()
	 */
	Collection<MinesweeperField> getInvertedNeighbors();
	
	
	/**
	 * Adds a neighbor to the neighbor-collection
	 * @param field {@link MinesweeperField} to add.
	 */
	void addNeighbor(MinesweeperField field);

	/**
	 * Adds a neighbor to the neighbor-collection relative to the current position
	 * @param dx X-position relative to current field to add
	 * @param dy Y-position relative to current field to add
	 */
	void addNeighbor(int dx, int dy);

	/**
	 * Remove a neighbor from the neighbor-collection.
	 * @param field {@link MinesweeperField} to remove.
	 * @return True if the field was removed, false if it did not exist in the neighbor-collection.
	 */
	boolean removeNeighbor(MinesweeperField field);
	
	/**
	 * @return The map ({@link MinesweeperGame}) to where this field belongs.
	 */
	MinesweeperMap getMap();
	
	/**
	 * Re-count value for field, will not modify any metadata.
	 */
	void init();
	
	/**
	 * 
	 * @return True if this field contains a mine (whether it has been clicked or not)
	 */
	boolean isMine();
	/**
	 * 
	 * @return
	 * -1 = Negative mine
	 *  0 = No mine
	 *  1 = Normal mine
	 *  2 = This position counts as two mines.
	 */
	int getMineValue();
	/**
	 * Sets this field to contain a mine or not
	 * @param mine
	 * @see #isMine()
	 */
	void setMine(boolean mine);
	
	/**
	 * How many mines are neighbors to this field.<br>
	 * Does not take into consideration if this field is a mine or not. Always returns the number of neighboring mines.
	 * 
	 * @return Number of neighboring mines
	 */
	int getValue();
	
	/**
	 * Sets the value for how many mines are neighbors to this field. Can be used to set false values on the map.
	 * @param value
	 */
	void setValue(int value);
	
	/**
	 * @return Gets the position of this field's coordinate in hexadecimal form.<br>
	 * For example, the field on position (7, 10) will return "7a"
	 */
	String getCoordinate();
	/**
	 * Sets this field to displayed and clicked by the player specified, automatically giving the player score if this field is a mine.
	 * 
	 * This method can be used by plugins.
	 * @param player The player who "owns" the field. A value of null will mark this field as inactivated. (Not displayed)
	 * @see #activate()
	 */
	void activate(MinesweeperPlayingPlayer player);
	/**
	 * Sets this field to displayed and clicked by the player who made the move.<br>
	 * <b>Does not expand.</b><br>
	 * This method should be the one used by weapons, not by plugins.
	 *  
	 * @param moveInfo
	 */
	void activate(MinesweeperMove moveInfo);
	/**
	 * Sets this field to displayed, without specifying a player.<br>
	 * 
	 * This method can be used by plugins.
	 * @see #activate(MinesweeperPlayingPlayer)
	 */
	void activate();
	/**
	 * Sets this field as not-clicked.
	 */
	void inactivate();
	
	/**
	 * Get a {@link MinesweeperField} relative to this position.
	 * @param deltaX X-offset
	 * @param deltaY Y-offset
	 * @return The {@link MinesweeperField} that is at position (this.getX() + deltaX, this.getY() + deltaY)
	 */
	MinesweeperField getRelativePosition(int deltaX, int deltaY);
	/**
	 * Copy the attributes from another field to this field
	 * @param position The position to copy from
	 */
	void copyFrom(MinesweeperField position);

	void setBlocked(boolean blocked);
	/**
	 * Find out whether this field is unclicked, clicked, blocked, or something else...
	 * @return A constant defined in the {@link MinesweeperField} class
	 */
	@Deprecated
	int getFieldType();
	FieldType getFieldStyle();

	boolean isBlocked();
	
	void sendInfoToPlayers();
	void sendInfoToPlayer(MinesweeperPlayingPlayer player);
}
