package net.zomis.minesweeper.api;

import net.zomis.minesweeper.game.MinesweeperField;
import net.zomis.minesweeper.game.MinesweeperGame;
import net.zomis.minesweeper.game.MinesweeperMap;
import net.zomis.minesweeper.game.MinesweeperPlayingPlayer;

public interface MinesweeperPlayer {
	/**
	 * Get the displayname of the player.
	 * @return A String of how the player will be displayed in clients to make it clickable.
	 */
	String getDisplayName();
	/**
	 * Get the name of the player
	 * @return The name of the player
	 */
	String getName();
	/**
	 * @return The {@link MinesweeperGame} in which the player is currently playing or observing, if any
	 */
	MinesweeperGame getGame();
	/**
	 * @return The {@link MinesweeperGame} in which the player is currently playing, if any
	 */
	MinesweeperGame getPlayingGame();
	/**
	 * @return The game id for the {@link MinesweeperGame} in which the player is currently playing
	 * Same as
	 * 	if (getPlayingGame() == null) return null;
	 *  else return getPlayingGame().getGameID();
	 */
	Integer getGameID();
	/**
	 * @return The {@link MinesweeperGame} in which the player is currently observing, if any
	 */
	MinesweeperGame getObservingGame();
	/**
	 * @return The player's current rating.
	 */
	Double getRating();
	/**
	 * Checks if the player is allowed to use a command.
	 * 
	 * @param command The command to check. Without the beginning slash. One word only
	 * @return True if the player is allowed to use the command, false otherwise
	 */
	boolean hasAccessToCommand(String command);
	
	@Deprecated
	boolean hasPermission(int permission);
	/**
	 * @return True if the player is an AI representation. False otherwise
	 */
	boolean isAI();
	/**
	 * @return True if the player is a guest. That is, the player has not signed up for an account
	 */
	boolean isGuest();
	/**
	 * @return True if the player is either playing or observing a game
	 */
	boolean isInGame();
	/**
	 * @return True if the player is observing a game
	 */
	boolean isObserving();
	/**
	 * @return True if the player is currently online
	 */
	boolean isOnline();
	/**
	 * @return True if the player is online and is a real player.
	 */
	boolean isRealPlayer();
	/**
	 * @return True only for the player object associated with the server.
	 */
	boolean isServer();
	/**
	 * @return True for the short moment where a player is connected and has not logged in yet.
	 */
	boolean isTemporaryPlayer();
	/**
	 * Sends a chat-message to this player.
	 * 
	 * @param message Message to send
	 * @return True if the message was sent
	 */
	boolean sendChat(String message);
	void sendChatBy(String message, MinesweeperPlayer from);
	void sendChatBy(String message, MinesweeperPlayer from, Integer chatColor);
	boolean sendChat(String message, Integer chatColor);
	
	/**
	 * Shows an image to the user on the field specified
	 * @param field The field at which to show the image
	 * @param url The URL of an image to show
	 */
	void sendFieldImage(MinesweeperField field, String url);
	
	/**
	 * @return The global {@link MinesweeperServer} object
	 */
	MinesweeperServer getServer();
	/**
	 * @return Metadata associated with this object
	 */
	Metadata getMetadata();
	/**
	 * Sends information about a field based on the current state of the field
	 * 
	 * Always sends true information
	 * @param field Field to send information about
	 * @param player Player which makes the move, if any
	 * @param highlight True if the player's associated highlight-square should be placed at the position
	 * @throws IllegalArgumentException if field is null or if field.getMap() returns null
	 */
	void sendFieldInfo(MinesweeperField field, MinesweeperPlayingPlayer player, boolean highlight);
	
	/**
	 * Sends an alert message to the player that will be displayed in a dialog box 
	 * @param alertMessage The message to send
	 */
	void sendAlert(String alertMessage);
	
	/**
	 * Check if the player is using the Android client
	 * @return True if the player is using the Android client
	 */
	boolean isClientAndroid();
	/**
	 * Check if the player is using the Flash client
	 * @return True if the player is using the Flash client
	 */
	boolean isClientFlash();

    @Deprecated
	boolean hasMineCoins(int amount);
    @Deprecated
	int getMineCoins();
    @Deprecated
	boolean addMineCoins(int amount);
	
	Integer getChatColor();
	void sendFieldInfo(MinesweeperField fieldPosition, FieldInfo fieldInfo);
	@Deprecated
	void sendFieldInfo(MinesweeperField fieldPosition, MinesweeperPlayingPlayer player, int fieldType, boolean isMine, int value, boolean highlight);
	Invite createInvitation(String[] plugins);
	
	void performCommand(String command);
	boolean hasPermission(String requiredPermission);
	String getFacebookUsername();
	int getUserId();
	UserClient getClient();
	MinesweeperMap getMap();
	void refreshMapView();
}
