package net.zomis.minesweeper.weapons;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;

import net.zomis.minesweeper.game.MinesweeperField;
import net.zomis.minesweeper.game.MinesweeperMove;
import net.zomis.minesweeper.game.MinesweeperPlayingPlayer;

public abstract class BaseWeapon implements MinesweeperWeapon {

	@Override
	public abstract boolean canUse(MinesweeperPlayingPlayer user);

	@Override
	public boolean canUseOn(MinesweeperPlayingPlayer user, MinesweeperPlayingPlayer target) {
		return false;
	}

	@Override
	public boolean use(MinesweeperPlayingPlayer user) {
		throw new RuntimeException("Illegal weapon use");
	}

	public interface RecursiveInterface {
		public boolean performAdd(MinesweeperField from, MinesweeperField to);
		public boolean performRecursive(MinesweeperField from, MinesweeperField to);
		public Collection<MinesweeperField> getRecursiveFields(MinesweeperField field);
	}
	/**
	 * Add fields to a collection recursively.
	 *  
	 * @param collection The collection where to add the added fields
	 * @param field Field to start checking. This field will not be added to collection from this method, if you want it added to the collection you must do that before or after this method.
	 * @param recursiveCheck Callback to determine which fields that should be scanned and/or added to the result.
	 */
	protected void recursiveAdd(Collection<MinesweeperField> collection, MinesweeperField field, RecursiveInterface recursiveCheck) {
		this.recursiveAdd(collection, field, recursiveCheck, new HashSet<MinesweeperField>());
	}
	
	private void recursiveAdd(Collection<MinesweeperField> collection, MinesweeperField field, RecursiveInterface recursiveCheck, Collection<MinesweeperField> recursiveChecked) {
		if (recursiveChecked.contains(field)) return;
		recursiveChecked.add(field);
		
		if (field == null) return; // Fix for NPE. 
		
		for (MinesweeperField mf : recursiveCheck.getRecursiveFields(field)) {
			if (mf == null) continue; // Fix for NPE.
			
			if (recursiveCheck.performAdd(field, mf)) {
				collection.add(mf);
			}
			if (recursiveCheck.performRecursive(field, mf)) {
				this.recursiveAdd(collection, mf, recursiveCheck, recursiveChecked);
			}
		}
	}
	/**
	 * Provided as an easy to use method for opening fields the normal way, with expander.
	 * @param field The field to open
	 * @param move Move information that will be passed to the fields.
	 */
	protected final void expanderOpen(MinesweeperField field, MinesweeperMove move) {
		field.activate(move);
		if (!field.isMine()) {
			if (field.getValue() == 0) {
				for (MinesweeperField f : field.getNeighbors())
				if (!f.isClicked()) {
					this.expanderOpen(f, move); // it's always the same move
				}
			}
		}
	}

	protected final Collection<MinesweeperField> getFieldsWithinRange(MinesweeperField field, int range) {
		List<MinesweeperField> result = new ArrayList<MinesweeperField>();
		
		for (int x = -range; x <= range; x++) {
			for (int y = -range; y <= range; y++) {
				MinesweeperField mf = field.getMap().getPosition(field.getX() + x, field.getY() + y);
				if (mf != null) result.add(mf);
			}
		}
		
		return result;
	}
	
	@Override
	public boolean useOn(MinesweeperPlayingPlayer user, MinesweeperPlayingPlayer target) {
		return false;
	}

	@Override
	public abstract String getWeaponType();

	@Override
	public abstract boolean canUseAt(MinesweeperPlayingPlayer user, MinesweeperField field);

	@Override
	public abstract Collection<MinesweeperField> getFieldsAffected(MinesweeperPlayingPlayer user, MinesweeperField field);

	@Override
	public abstract boolean useAt(MinesweeperMove moveInfo);
	
	@Override
	public String toString() {
		return this.getClass().getSimpleName();
	}
}