package net.zomis.minesweeper.api;

import net.zomis.minesweeper.game.MinesweeperMap;
import net.zomis.minesweeper.game.MinesweeperMove;
import net.zomis.minesweeper.game.MinesweeperPlayingPlayer;

public abstract class MinesweeperAI {
	private final MinesweeperPlayingPlayer player;
	private boolean hidden = false;
	private AIHelper	helper;
	
	public MinesweeperAI(MinesweeperPlayingPlayer player) {
		if (player == null) throw new NullPointerException("Cannot create a MinesweeperAI without a playing player specified.");
		this.player = player;
	}
	
	public final boolean isHidden() {
		return this.hidden;
	}
	protected final void hide() {
		this.hidden  = true;
	}

	/**
	 * The main AI method which determines how the AI will play
	 * 
	 * @return A new instance of {@link MinesweeperMove} to indicate which move the AI prefer to do.
	 * Check if your move is possible before making it with {@link MinesweeperMove}.isAllowed(), otherwise your AI will look stupid for trying to make an illegal move.
	 */
	public abstract MinesweeperMove play();
	
	/**
	 * @param invite The {@link Invite} object
	 * @return True if the AI agrees to play against the player specified
	 */
	public abstract boolean respondToInvite(Invite invite);
	/**
	 * @return True if the AI agrees to draw the game, false otherwise
	 */
	public abstract boolean agreeDraw();

	public final MinesweeperPlayer getPlayer() {
		return this.player.getPlayer();
	}
	public final MinesweeperMap getMap() {
		return this.player.getMap();
	}
	public final MinesweeperPlayingPlayer getPlayingPlayer() {
		return this.player;
	}

	public final void init(AIHelper aiHelper) {
		if (this.helper != null) throw new IllegalStateException("AI Helper is already set.");
		this.helper = aiHelper;
	}
	
	public final void sendChatMessage(String message) {
		if (this.helper != null) {
			this.helper.sendMessage(this.getPlayingPlayer(), message);
		}
	}
	
	public String getName() {
		return this.getClass().getSimpleName();
	}
}
