package net.zomis.minesweeper.game;

import java.util.List;

import net.zomis.minesweeper.api.Metadata;
import net.zomis.minesweeper.api.MinesweeperPlayer;
import net.zomis.minesweeper.api.MinesweeperPlugin;
import net.zomis.minesweeper.api.MinesweeperServer;
/**
 * A Game of Minesweeper Flags Extreme
 */
public interface MinesweeperGame extends MinesweeperMap {
	
	public int getGameID();
	
	/**
	 * Get all players currently observing this game
	 * @return A list of all the players currently observing players
	 */
	List<MinesweeperPlayer> getObservers();
	/**
	 * Get a list of players and observers combined
	 * @return A list of all players and observers combined.
	 */
	List<MinesweeperPlayer> getSendPlayers();
	/**
	 * Retrieve a list of all players that are playing in this game
	 * @return A list of players that are playing in the game
	 */
	List<MinesweeperPlayer> getPlayers();
	/**
	 * Check if this game has the specified player
	 * @param player The player to check for
	 * @return True if the specified player is a player in this game, false otherwise.
	 */
	boolean hasPlayer(MinesweeperPlayer player);
	/**
	 * Check if the specified player is an observer in this game
	 * @param player The player to check for
	 * @return True if the specified player is observing this game, false otherwise.
	 */
	boolean hasObserver(MinesweeperPlayer player);
	
	/**
	 * Check if this game only contains one human player and at least one AI players.
	 * @return True if this game contains one human player and at least one AI.
	 */
	boolean isPlayerVsAI();
	/**
	 * Check if this game is stored in the global database
	 * @return True if this game is stored in the database, false otherwise
	 */
	boolean isStoredInDatabase();
	
	/**
	 * Get the corresponding {@link MinesweeperPlayingPlayer} for a {@link MinesweeperPlayer}
	 * @param player The player to get the playing player for
	 * @return A {@link MinesweeperPlayingPlayer} that belongs to this game and the specified {@link MinesweeperPlayer}, or null if none was found.
	 */
	MinesweeperPlayingPlayer getPlayingPlayer(MinesweeperPlayer player);
	/**
	 * Get the metadata for this game
	 * @return {@link Metadata} object belonging to this game.
	 */
	Metadata getMetadata();
	/**
	 * Checks if the specified plugin is enabled in this game.
	 * @param plugin The plugin to check
	 * @return True if the plugin is enabled in this game, false otherwise.
	 */
	boolean hasPlugin(MinesweeperPlugin plugin);
	/**
	 * Get the global {@link MinesweeperServer} object
	 * @return The server on which this game is played
	 */
	MinesweeperServer getServer();
	/**
	 * Send a chat-message to the players and observers in this game.
	 * @param message The message to send.
	 */
	void sendChatMessage(String message);
	/**
	 * End this game.
	 */
	void endGame();
	boolean addObserver(MinesweeperPlayer user);
	void removeObserver(MinesweeperPlayer user);
	MoveAllowedState performClicks(String clicks);
	void sendChatMessage(String message, Integer color);

	void addTag(GameTag tag);

	List<GameTag> getTags();
	void setAIDelay(int delay);
	int getAIDelay();

	void sendChatMessage(String fromName, String message);
	void sendChatMessage(String fromName, String message, Integer color);
	void sendChatMessage(MinesweeperPlayer from, String message);

	long getTimeLastAction();

}