package net.zomis.minesweeper.game;

import java.util.List;
import java.util.Random;

import net.zomis.minesweeper.api.MapFactory;
import net.zomis.minesweeper.api.MinesweeperPlugin;
import net.zomis.minesweeper.events.game.CustomEvent;
import net.zomis.minesweeper.weapons.MinesweeperWeapon;


public interface MinesweeperMap {
	/**
	 * Retrieve the number of moves that has been made in this game
	 * @return The number of moves that has been done in this game
	 */
	 int getClickCount();
	/**
	 * Check if this game has ended
	 * @return True if the winners of this game has been declared, false otherwise.
	 */
	 boolean isGameOver();
	
	/**
	 * @return The number of {@link MinesweeperField}s in X-axis
	 */
	 int getFieldWidth();
	
	/**
	 * @return The number of {@link MinesweeperField}s in Y-axis
	 */
	 int getFieldHeight();

	/**
	 * Get the field at a specific location on the map.
	 * @param x X-position of the field
	 * @param y Y-position of the field
	 * @return The {@link MinesweeperField} at position (x, y)
	 */
	 MinesweeperField getPosition(int x, int y);
	/**
	 * Get the total number of mines in this game
	 * @return The total number of mines in this game
	 */
	 int getMinesCount();
	/**
	 * Get the number of non-revealed mines
	 * @return The total number of mines subtracted by the total number of found mines by all players. 
	 */
	 int getMinesLeft();
	/**
	 * Retrieve a list of all non-revealed fields.
	 * @return A list of all non-revealed fields.
	 */
	 List<MinesweeperField> getAllUnclickedFields();
	/**
	 * Retrieve a list of all fields.
	 * @return A list of all fields.
	 */
	 List<MinesweeperField> getAllFields();
	/**
	 * Get the player who's turn it is
	 * @return The player who's turn it is
	 */
	 MinesweeperPlayingPlayer getCurrentPlayer();
	/**
	 * Get the index of who's turn it is
	 * @return The index of the current player
	 */
	 int getPlayerTurn();
	/**
	 * Change the turn to the next player
	 */
	 void nextTurn();
	/**
	 * Get an iteration for all {@link MinesweeperField}s in this map
	 * @return An iteration of all {@link MinesweeperField}s in this map, useful in for-loops:
	 * <pre>for (MinesweeperField field : {@link MinesweeperMap}.getIteration()) {
	 *   ...
	 *}</pre>
	 */
	 Iterable<MinesweeperField> getIteration();
	/**
	 * Get a list of the playing players in this game
	 * @return A list of {@link MinesweeperPlayingPlayer} objects.
	 */
	 List<MinesweeperPlayingPlayer> getPlayingPlayers();
	/**
	 * Generate this game. Resets everything, clears the map, restarts the game, alerts all plugins
	 */
	 void generate();
	 @Deprecated
	 void generate(Random randomizer);
	 void generate(int mines, Random randomizer);
	 void generate(int mines);
	/**
	 * Check if this game allows all players to play at once
	 * @return True if all players can play simultaneously. Defaults to false
	 */
	 boolean isActionMode();
	/**
	 * Sets if action mode should be on or off (Default is off)
	 * @param actionMode True to allow players to play simultaneously, false to disable.
	 */
	 void setActionMode(boolean actionMode);
	/**
	 * Save this map to a String representation
	 * @return A String representation of this map
	 * @see #loadMap(String)
	 */
	 String saveMap();
	/**
	 * Save this map to a String representation
	 * @return A String representation of this map
	 * @see #loadMap(String)
	 */
	 String[] saveMapMultiline();
	/**
	 * Load game state from a String that has been saved with {@link #saveMap()}
	 * @param data The String representation to load
	 * @see #saveMap()
	 * @return The same map
	 */
	MinesweeperMap loadMap(String data);
	/**
	 * Clear the entire map. No blocked fields, no mines
	 */
	void clear();
	/**
	 * Get a list of all the moves that has been made in this game
	 * @return A list of all the moves that has been made in this game.
	 */
	List<MinesweeperMove> getMoveHistory();
	/**
	 * Copy the data from another game to this game 
	 * @param game The game to copy from
	 */
	@Deprecated
	void copyFrom(MinesweeperMap game);
	
	/**
	 * Perform an AI-call. This only has effect if this is an AI-test game.
	 */
	void callAI();
	void setCurrentPlayer(MinesweeperPlayingPlayer player);
	void setCurrentPlayerTurn(int playerIndex);
	
	void initFields();
	List<MinesweeperPlayingPlayer> getRemainingPlayers();
	void sendUpdateToPlayers();
	boolean isDraw();
	void setMinePositions(String minepos, int charsPerPos);
	void setMinePositions(String minepos);
	int getPlayersLeftCount();
	MinesweeperMove createMove(MinesweeperPlayingPlayer player, MinesweeperWeapon weapon, MinesweeperField field);
	
	boolean hasPlugin(MinesweeperPlugin plugin);
	
	void addPlugin(MinesweeperPlugin plugin);
	
	/**
	 * A list containing the {@link MinesweeperPlugin} that is enabled in this game
	 * @return A list of the activated plugins in this game.
	 */
	List<MinesweeperPlugin> getActivePlugins();
	
	MapFactory getMapFactory();
	
	MoveAllowedState performMove(MinesweeperMove move);
	void setMinesCount(int minesCount);
	MinesweeperReplay createReplay();
	MinesweeperField getPosition(MinesweeperField field);
	CustomEvent executeCustomEvent(CustomEvent customEvent);
}
