package net.zomis.minesweeper.analyze.utils;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;

import net.zomis.minesweeper.analyze.AnalyzeResult;
import net.zomis.minesweeper.analyze.impl.MineprobabilityAnalyze;
import net.zomis.minesweeper.game.MinesweeperField;
import net.zomis.minesweeper.game.MinesweeperMap;
import net.zomis.minesweeper.weapons.MinesweeperWeapon;
import net.zomis.minesweeper.weapons.classic.ClickWeapon;

public class OpenFieldAnalyzer {
	private final MinesweeperMap map;

	public OpenFieldAnalyzer(MinesweeperMap map) {
		this.map = map;
	}
	
	private static Random random = new Random();
	
	public Map<MinesweeperField, Double> analyzePossibleOpenFields(int totalTime) {
		Map<MinesweeperField, Double> result = new HashMap<MinesweeperField, Double>();
		List<MinesweeperField> unclicked = map.getAllUnclickedFields();
//		logger.info("Unclicked: " + unclicked.size());
		if (unclicked.isEmpty()) return result;
		final int tests = totalTime / unclicked.size();
//		logger.info("Tests: " + tests);
		String str = map.saveMap();
		for (MinesweeperField ff : unclicked) {
			if (!ff.isClicked()) {
//				logger.info("Testing: " + ff);
				double dbl = getExpected100fromOpenField(str, ff, tests, false);
				if (dbl > 0)
					result.put(ff, dbl);
			}
		}
		
		return result;
	}
	
//	static MineprobabilityAnalyze prepareOpenFieldAnalyze(MinesweeperMap map, ) {
//		Create a method so that no new map needs to be created every iteration in analyzePossibleOpenFields.
//	}
	
	public static double getExpected100fromOpenField(final String map, MinesweeperField field, final int tests, final boolean testsAsTime) {
//		logger.info("Map: " + map);
		
		MinesweeperMap tempMap = field.getMap().getMapFactory().withTwoPlayers().loadFrom(map).map();
		field = tempMap.getPosition(field.getX(), field.getY());
		
		for (MinesweeperField ff : field.getNeighbors()) {
			if (ff.isClicked() && ff.isMine()) return Integer.MIN_VALUE;
		}
		
		MineprobabilityAnalyze analyze = new MineprobabilityAnalyze(tempMap);
		analyze.addRule(MineprobabilityAnalyze.ruleFromField(field, 0));
		analyze.addRule(MineprobabilityAnalyze.ruleForField(field, false));
		AnalyzeResult<MinesweeperField> result = analyze.solve();
		
		clearHiddenMines(tempMap);
		
		if (result.getTotal() == 0) {
			return Integer.MIN_VALUE;
		}
		
		int start100 = MineprobHelper.find100(result);
		int total = 0;
		
		field = tempMap.getPosition(field.getX(), field.getY());
		ClickWeapon weapon = new ClickWeapon();
		
		int totalTests = 0;
		if (!testsAsTime) {
			for (int i = 0; i < tests; i++) {
//				logger.info(field + " Loop " + i);
				totalTests++;
				tempMap.loadMap(map);
				total += getExpected100fromOpenFieldIteration(tempMap, result, random, field, weapon, start100);
			}
		}
		else {
			long startTime = System.currentTimeMillis();
			while (System.currentTimeMillis() - startTime < tests) {
//				logger.info(field + " Loop " + totalTests);
				totalTests++;
				tempMap.loadMap(map);
				total += getExpected100fromOpenFieldIteration(tempMap, result, random, field, weapon, start100);
			}
		}
		return 1.0 * total / totalTests;
	}
	private static void clearHiddenMines(MinesweeperMap tempMap) {
		for (MinesweeperField ff : tempMap.getIteration())
			if (!ff.isClicked() && ff.isMine())
				ff.setMine(false);
		
		tempMap.saveMap();
	}

	private static int getExpected100fromOpenFieldIteration(MinesweeperMap tempMap, 
			AnalyzeResult<MinesweeperField> analyze, Random random, MinesweeperField field, MinesweeperWeapon weapon, int start100) {
//		List<MinesweeperField> randomFields = fastRegenerate(analyze, random);
		MineprobHelper.regenerate(tempMap, analyze, random);
//		if (randomFields == null) throw new AssertionError("Regenerate returned null");
		
		if (field.isClicked()) throw new AssertionError(field.toString() + " is already clicked. " + field.getMap().saveMap());
		if (field.isMine()) throw new AssertionError(field.toString() + " is a mine. " + field.getMap().saveMap());
		if (field.getValue() != 0) throw new AssertionError(field.toString() + " is a " + field.getValue() + ". " + field.getMap().saveMap());
		weapon.useAt(field.getMap().createMove(null, weapon, field));
		
		MineprobabilityAnalyze analyze2 = new MineprobabilityAnalyze(tempMap);
		AnalyzeResult<MinesweeperField> result2 = analyze2.solve();
//		clearMines(randomFields);
//		logger.info(tempMap.saveMap());
		return MineprobHelper.find100(result2) - start100;
	}
/*	private static void clearMines(List<MinesweeperField> fields) {
		for (MinesweeperField ff : fields) ff.setMine(false);
	}
	private static List<MinesweeperField> fastRegenerate(MineprobabilityAnalyze analyze, Random random) {
		List<MinesweeperField> recalc = new ArrayList<MinesweeperField>();
		List<MinesweeperField> chosen = analyze.randomSolution(random);
		for (MinesweeperField ff : chosen) {
			recalc.addAll(ff.getNeighbors());
			ff.setMine(true);
		}
		
//		for (MinesweeperField ff : recalc)
//		if (!ff.isClicked())
//			ff.init();
		recalc.get(0).getMap().initFields();
		
		return chosen;
	}*/
}
