package net.zomis.minesweeper.ais.utils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;

import net.zomis.UtilZomisUtils;
import net.zomis.minesweeper.aiscore.AvoidReveal50;
import net.zomis.minesweeper.analyze.FieldGroup;
import net.zomis.minesweeper.analyze.detail.ProbabilityKnowledge;
import net.zomis.minesweeper.analyze.impl.AnalyzeProvider;
import net.zomis.minesweeper.analyze.utils.MineprobHelper;
import net.zomis.minesweeper.analyze.utils.OpenFieldApproxer;
import net.zomis.minesweeper.game.MinesweeperField;
import net.zomis.minesweeper.game.MinesweeperMap;
import net.zomis.minesweeper.scores.AbstractScorer;

public class ChickenAnalyze {

	private final AnalyzeProvider	analyze;
	private final List<OpenField> openFields;
	private int	extraSafe = 0;
	private final Set<MinesweeperField> unsafeReveal;
	private final Set<MinesweeperField> unsafe;

	public int getSafeCount() {
		return this.extraSafe;
	}
	public int getUnsafeCount() {
		return this.unsafe.size();
	}
	public Set<MinesweeperField> getUnsafe() {
		return new HashSet<MinesweeperField>(this.unsafe);
	}
	public List<OpenField> getOpenFields() {
		return new ArrayList<OpenField>(this.openFields);
	}
	public Set<MinesweeperField> getUnsafeReveal() {
		return new HashSet<MinesweeperField>(this.unsafeReveal);
	}
	
	public ChickenAnalyze(MinesweeperMap map, AnalyzeProvider analyze) {
//		this.map = map;
		this.analyze = analyze;
		this.openFields = new ArrayList<OpenField>();
		this.unsafeReveal = new HashSet<MinesweeperField>();
		this.unsafe = new HashSet<MinesweeperField>();
	}
	
	public ChickenAnalyze analyze() {
		OpenFieldApproxer approxer = new OpenFieldApproxer();
		
		AbstractScorer scorer = new AvoidReveal50();
		
		Set<ProbabilityKnowledge<MinesweeperField>> fieldOfInterest = new HashSet<ProbabilityKnowledge<MinesweeperField>>();
		for (ProbabilityKnowledge<MinesweeperField> know : analyze.getAllKnowledge()) {
			double ofProb = know.getProbabilities()[0];
			if (ofProb == 1.0) {
				if (approxer.expectedFrom(analyze, know.getField()) == 0)
					addOpenField(know);
			}
			else if (ofProb == 0) {
				if (scorer.getScoreFor(know.getField(), know, null) == 0)
					fieldOfInterest.add(know);
			}
		}
		
		for (ProbabilityKnowledge<MinesweeperField> data : fieldOfInterest) {
			if (MineprobHelper.getCertainValue(data) != null) {
				if (!existsInOpenField(data.getField())) {
					this.extraSafe++;
				}
			}
			else if (data.getMineProbability() == 0.0) {
				this.addUnsafe(data);
				// Unsafe click, check the neighbors of the knowledge.
			}
		}
		// TODO: Add analyze of untaken safe 50%-mines??
		
		/* Count odd-sized open fields
		 * Count even-sized open fields
		 * Count non-revealing fields --- do not count the ones close to one of the above
		 * Count small revealing fields (with risk of revealing 100%)
		 * 
		 * Check for common neighbors in small revealing fields
		 **/
		
		// _____112a2bb21a1-_____1b2234a__2_-11___1233b2__12b-_b2a212bb2___b2_-22323b332211__2_-b23b44b202a21b_1-__bb4bb324a223a1-__224b42aa211a32-____2b2122223_3b-___________aa4a_-__x__x__123a_aa2-xx_____2a3b22221-_______3a3_321__-x______a211aa1__-121212a2_____1__-__x_x__1________
		// x____a3b11111bb1-_1___2b222b11221-______122b211221-_xx___12b2223ba2-_x____2a421bb55b-__x___3ab3333bb2-x_x___b33bb11221-_x__122123421___-____1a1_1a3a311_-____1_____3a3b1_-________________-x___xx__________-__x______xx___xx-x_________x_____-_1_x_x_______x_x-___x_____x_x____
		// 23211111222b2222-baa11b__bb43b3ba-233221_a5b6a4a32-112a1__14baa4210-2a2221__2a54b100-b212b21___b21100-11_3b42aa_221000-___2ba22211b1000-___122___1233210-__________a2aa31-_x____x__1245b3a-__x_x_____1aa221-______x_x_1221__-x___x___________-_xx____x________-_________xx___x_
		
		
		return this;
	}

	private void addUnsafe(ProbabilityKnowledge<MinesweeperField> data) {
		this.unsafe.add(data.getField());
		this.unsafeReveal.addAll(getUnsafeRevealFor(data));
		
		
	}

	public static Set<MinesweeperField> getUnsafeRevealFor(ProbabilityKnowledge<MinesweeperField> data) {
		Set<MinesweeperField> result = new HashSet<MinesweeperField>();
		for (Entry<FieldGroup<MinesweeperField>, Integer> ee : data.getNeighbors().entrySet()) {
			if (ee.getKey().size() != ee.getValue()) { // Not neighbor with all of them
				Collection<MinesweeperField> neighbors = data.getField().getNeighbors();
				neighbors.retainAll(ee.getKey());
				result.addAll(neighbors);
			}
		}
		return result;
	}
	private void addOpenField(ProbabilityKnowledge<MinesweeperField> know) {
		if (existsInOpenField(know.getField())) return; // Each field will only be connected to one OpenField, because of the neighborset.
		
		OpenField of = OpenField.construct(this.analyze, know);
//		logger.info("OF ADDED: " + of);
		this.openFields.add(of);
	}

	private boolean existsInOpenField(MinesweeperField field) {
		for (OpenField of : this.openFields)
			if (of.hasField(field)) return true;
		
		return false;
	}

	@Override
	public String toString() {
		return UtilZomisUtils.implode(", ", this.openFields) + " - extraSafe: " + this.extraSafe + " unsafe: " + this.unsafe 
				+ " unsafeReveal: " + UtilZomisUtils.implode(", ", this.unsafeReveal);
	}
	public int getSafeRevealedBy(AnalyzeProvider analyze, ProbabilityKnowledge<MinesweeperField> fieldData) {
		if (!this.unsafe.contains(fieldData.getField())) throw new IllegalArgumentException();
//		Logger.getLogger(getClass()).info(fieldData.getField() + " would reveal what?");
		
		int i = 0;
		Set<MinesweeperField> myRevealOriginal = getUnsafeRevealFor(fieldData);
//		Logger.getLogger(getClass()).info(fieldData.getField() + " unsafe reveal is " + myRevealOriginal);
		
		for (MinesweeperField ff : this.unsafe) {
			if (ff == fieldData.getField()) continue;
			
			Set<MinesweeperField> ffReveal = ChickenAnalyze.getUnsafeRevealFor(analyze.getKnowledgeFor(ff));
//			Logger.getLogger(getClass()).info("ffReveal is " + ff + ", " + ffReveal);
			Set<MinesweeperField> myReveal = new HashSet<MinesweeperField>(myRevealOriginal);
			ffReveal.removeAll(myReveal);
//			Logger.getLogger(getClass()).info("myReveal " + myReveal + ". ffReveal " + ffReveal);

			if (ffReveal.size() == 0) {
				i++;
			}
		}
		
		return i;
	}
}
