package net.zomis.plugin;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.zomis.mfe.plugin.ThreadInterruptCheck;
import net.zomis.minesweeper.ais.AI_Nightmare;
import net.zomis.minesweeper.ais.AI_Zomis;
import net.zomis.minesweeper.ais.StaticAISupplier;
import net.zomis.minesweeper.analyze.detail.ProbabilityKnowledge;
import net.zomis.minesweeper.api.MinesweeperPlayer;
import net.zomis.minesweeper.api.MinesweeperPlugin;
import net.zomis.minesweeper.classic.DefaultExpanderRule;
import net.zomis.minesweeper.classic.StandardNeighbors;
import net.zomis.minesweeper.classic.StandardWeapons;
import net.zomis.minesweeper.events.Command;
import net.zomis.minesweeper.events.EventRegister;
import net.zomis.minesweeper.events.EventRegistrator;
import net.zomis.minesweeper.events.game.*;
import net.zomis.minesweeper.events.player.PlayerCommandEvent;
import net.zomis.minesweeper.functional.Consumer;
import net.zomis.minesweeper.game.MinesweeperField;
import net.zomis.minesweeper.game.MinesweeperGame;
import net.zomis.minesweeper.game.MinesweeperMap;
import net.zomis.minesweeper.game.MinesweeperMove;
import net.zomis.minesweeper.game.MinesweeperPlayingPlayer;
import net.zomis.minesweeper.scores.AbstractScorer;
import net.zomis.minesweeper.scores.FieldScore;
import net.zomis.minesweeper.scores.FieldScores;
import net.zomis.minesweeper.weapons.MinesweeperWeapon;

public class FeedbackPlugin extends MinesweeperPlugin implements EventRegistrator {
	private static String SENDER = FeedbackPlugin.class.getSimpleName();

	@Override
	public void registerEvents(EventRegister events) {
		events.registerListener(PlayerEliminatedEvent.class, new Consumer<PlayerEliminatedEvent>() {
			@Override
			public void accept(PlayerEliminatedEvent playerEliminatedEvent) {
				onPlayerEliminated(playerEliminatedEvent);
			}
		});
		events.registerListener(PlayerMoveEvent.class, new Consumer<PlayerMoveEvent>() {
			@Override
			public void accept(PlayerMoveEvent playerMoveEvent) {
				onMove(playerMoveEvent);
			}
		});
		events.registerListener(PlayerAfterMoveEvent.class, new DefaultExpanderRule());
		// events.registerListener(GameEndedEvent.class, new EndShowMines());
		events.registerListener(GamePreGenerateEvent.class, StandardNeighbors.classicNeighbors);
		events.registerListener(GameInitEvent.class, new StandardWeapons());
	}

	private class FeedbackMeta {
		private Map<MinesweeperPlayingPlayer, List<Double>> list = new HashMap<MinesweeperPlayingPlayer, List<Double>>();
		
		private void addMove(MinesweeperPlayingPlayer player, Double score) {
			if (!list.containsKey(player)) {
				List<Double> l = new ArrayList<Double>();
				list.put(player, l);
			}
			list.get(player).add(score);
		}

		public double calcAverage(MinesweeperPlayingPlayer pp) {
			List<Double> list = this.list.get(pp);
			if (list == null) return Integer.MIN_VALUE;
			if (list.size() == 0) return 0;
			double total = 0;
			for (Double d : list) total += d;
			return total / list.size();
		}
		
	}
	private FeedbackMeta getMeta(MinesweeperGame game) {
		FeedbackMeta meta = game.getMetadata().getClass(FeedbackMeta.class);
		if (meta == null) return game.getMetadata().addClass(new FeedbackMeta());
		else return meta;
	}
	
	public void onPlayerEliminated(PlayerEliminatedEvent event) {
		if (event.getGame() == null) return;
		event.getGame().sendChatMessage(SENDER, "Average for " + event.getPlayingPlayer() + ": " + getMeta(event.getGame()).calcAverage(event.getPlayingPlayer()));
	}

	private class AsyncFeedback implements Runnable {
		private MinesweeperGame	game;
		private AI_Zomis zomis;
		private MinesweeperMove	move;
		private MinesweeperMap	map;

		public AsyncFeedback(MinesweeperGame game, AI_Zomis ai, MinesweeperMove move) {
			this.game = game;
			this.map = game;
			this.zomis = ai;
			this.move = move;
		}

		@Override
		public void run() {
			if (!this.move.getMoveAllowedState().isOK()) {
				game.sendChatMessage(FeedbackPlugin.this.getSimpleName(), "Move " + this.move + " not allowed", null);
				return;
			}
            MinesweeperPlayer player = game.getCurrentPlayer().getPlayer();
			if (zomis == null) {
				zomis = new AI_Nightmare(new ThreadInterruptCheck());
			}
			game.sendChatMessage(player, "Performing.");
            MinesweeperPlayingPlayer pp = game.getCurrentPlayer();
			AI_Zomis ai = new AI_Nightmare(new ThreadInterruptCheck());
			FieldScores scores = ai.createScoreProvider(pp).analyzeAndScore(move.getWeapon(), false);
			scores.rankScores();
			
			MinesweeperField fieldPlayed = move.getMovePosition();
			FieldScore score = scores.getScoreFor(fieldPlayed);
			
			int rank = score.getRank();
			int ranklength = scores.getRankCount();
			double normScore = scores.getNormalized(fieldPlayed);
			if (normScore == -2) {
				normScore = -1;
				game.sendChatMessage(player, "AI did not even thought about considering move " + move + ". Setting score for the move to -1.");
			}
			Integer color = null;
			if (normScore < 0.9) color = 0xff8080;
			else color = 0x00ff00;

            game.sendChatMessage(SENDER, zomis.getName() + ": " + move.getPlayer().getName() + " (" + move.getPlayer().getScore() + ") used "
                    + move.getWeapon() + " at " + move.getField() + ". Score " + normScore + ". Rank " + rank + " / " + ranklength);

            getMeta(game).addMove(move.getPlayer(), normScore);
				
			this.sendFieldProbabilities(fieldPlayed, scores.getAnalyze().getKnowledgeFor(fieldPlayed));
		}
		private void sendFieldProbabilities(MinesweeperField field, ProbabilityKnowledge<MinesweeperField> data) {
			if (data == null) return;
			int min = -1;
			int max = 9;
			for (int i = 0; i < data.getProbabilities().length; i++) {
				if (data.getProbabilities()[i] <= 0) continue;
				max = i;
				if (min == -1) min = i;
			}
			game.sendChatMessage(SENDER, field + " had mine probability " + data.getMineProbability());
			if (min != -1)
				game.sendChatMessage(SENDER, field + " had probability of " + min + ": " + data.getProbabilities()[min]);
			if (max != 9)
				game.sendChatMessage(SENDER, field + " had probability of " + max + ": " + data.getProbabilities()[max]);
			
//			game.sendChatMessage(field + " is " + (field.isMine() ? "mine" : field.getValue()) + ", probability " + 
//				(field.isMine() ? data.getMineProbability() : data.getProbabilities()[field.getValue()]) + ". Possible values: " + min + "--" + max);
		}
		
	}
	
	public void onMove(PlayerMoveEvent event) {
		if (event.getServer() == null) return;
		
//		new Thread(new AsyncFeedback(event.getGame(), this.getAIZomis(event.getGame()), event.getMove())).start();
		new AsyncFeedback(event.getGame(), this.getAIZomis(event.getMap()), event.getMove()).run(); // TODO: Make FeedbackPlugin an asynctask. Need a snapshot of how the game looked like.
	}
	
	private AI_Zomis getAIZomis(MinesweeperMap game) {
		for (MinesweeperPlayingPlayer pp : game.getPlayingPlayers()) {
			if (pp.isAI() && pp.getAI() instanceof AI_Zomis) return (AI_Zomis) pp.getAI();
		}
		
		return null;
	}

//	@Event
//	public void checkForAI(GamePreStartEvent event) {
//		for (MinesweeperPlayingPlayer pp : event.getMap().getPlayingPlayers()) {
//			if (pp.getPlayer().isTemporaryPlayer()) return;
//
//			if (pp.isAI() && pp.getAI() instanceof AI_Zomis) {
//				return;
//			}
//		}
//
////		event.setCancelled(true);
//	}

	@Command(command = "feedback", help = "Get some feedback", requiredPermission = PlayerCommandEvent.PERMISSION_TRUSTED_USER)
	public void feedback(PlayerCommandEvent event) {
		MinesweeperPlayingPlayer pp = event.getMap().getCurrentPlayer();
		AI_Zomis ai = new AI_Nightmare(new ThreadInterruptCheck());
		MinesweeperWeapon weapon = pp.getWeapon(MinesweeperMove.STANDARD_CLICK);
		
		FieldScores scores = ai.createScoreProvider(pp).analyzeAndScore(weapon, false);
		MinesweeperField field = event.getMap().getPosition(event.getParameterInt(0, -1), event.getParameterInt(1, -1));
		ProbabilityKnowledge<MinesweeperField> data = scores.getAnalyze().getKnowledgeFor(field);
		event.getPlayer().sendChat("Checking field " + field);
		
		for (AbstractScorer scorer : ai.getConfig().getScorers().keySet()) {
			if (weapon.canUseAt(pp, field)) break;
			
			double score = scorer.getScoreFor(field, data, scores);
			
			if (score != 0.0) {
				event.getPlayer().sendChat(scorer.getName() + " gives " + score);
			}
			else event.getPlayer().sendChat(scorer.getName() + " score was zero");
		}
		event.getPlayer().sendChat("Done!");
		
	}
	
	@Override
	public void onEnable() {
		// this.registerListener(this);
		this.registerListener(new DefaultExpanderRule());
		// this.registerListener(new StandardNeighbors());
		// this.registerListener(new StandardWeapons());
	}

	@Override
	public void onDisable() {
	}

	@Override
	public boolean canBeChosenBy(MinesweeperPlayer user) {
		return true; // user.hasPermission(PlayerCommandEvent.PERMISSION_TRUSTED_USER);
	}

}
