package net.zomis.minesweeper.ais;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map.Entry;

import net.zomis.UtilZomisList;
import net.zomis.UtilZomisUtils;
import net.zomis.minesweeper.analyze.*;
import net.zomis.minesweeper.analyze.detail.DetailedResults;
import net.zomis.minesweeper.analyze.endgame.WinChanceTools;
import net.zomis.minesweeper.game.MinesweeperField;
import net.zomis.minesweeper.game.MinesweeperMap;
import net.zomis.minesweeper.game.MinesweeperMove;
import net.zomis.minesweeper.game.MinesweeperPlayingPlayer;

public class BombTools {
	private MinesweeperPlayingPlayer player;

	public BombTools(MinesweeperPlayingPlayer player) {
		this.player = player;
	}
	public static boolean isInterestingSolution(Solution<MinesweeperField> solution) {
		for (Entry<FieldGroup<MinesweeperField>, Integer> ee : solution.getSetGroupValues().entrySet()) {
			if (ee.getKey().size() - ee.getValue() != 1) { // check for 2a = 1, 3a = 2, 4a = 3... (commonly EV 0 situations)
				if (ee.getKey().size() <= 8) return true;
			}
		}
		return false;
	}
	
	public static double getBombWinPercent(AnalyzeResult<MinesweeperField> analyze, MinesweeperField bombLocation, int score) {
		int remaining = bombLocation.getMap().getMinesLeft();
		int needed = WinChanceTools.NEEDED_SCORE - score;
		
		List<MinesweeperField> adj = getBombAdjacents(bombLocation);
		UtilZomisList.filter(adj, new UtilZomisList.FilterInterface<MinesweeperField>() {
			@Override
			public boolean shouldKeep(MinesweeperField obj) {
				return !obj.isClicked();
			}
		});
		if (remaining > adj.size()) remaining = adj.size();
		
		double total = 0;
		
		for (int i = needed; i <= remaining; i++) {
			List<RuleConstraint<MinesweeperField>> list = new ArrayList<RuleConstraint<MinesweeperField>>();
			FieldRule<MinesweeperField> bombRule = new FieldRule<MinesweeperField>(bombLocation, adj, i);
			list.add(bombRule);
			double probI = analyze.getProbabilityOf(list);
//			logger.info("win% for " + bombLocation + ": " + i + " mines with probability of " + probI);
			total += probI;
		}
		return total;
	}
	
	public MinesweeperMove checkBomb(DetailedResults<MinesweeperField> detailedAnalyze) {
		double highest = 0;
		MinesweeperField highestField = null;
		for (MinesweeperField ff : player.getMap().getIteration()) {
			double bombprob = this.calcBombProb(ff, detailedAnalyze);
			if (bombprob > highest) {
				highest = bombprob;
				highestField = ff;
			}
		}
		
		if (player.getScore() + highest >= 25.5) {
			return player.createMove(MinesweeperMove.STANDARD_BOMB, highestField);
		}
		return null;
	}

	public double calcBombProb(MinesweeperField ff, DetailedResults<MinesweeperField> detailedAnalyze) {
		double total = 0;
		for (int xx = -2; xx <= 2; xx++) {
			for (int yy = -2; yy <= 2; yy++) {
				MinesweeperField relative = ff.getRelativePosition(xx, yy);
				if (relative == null) continue;
				if (relative.isClicked()) continue;
				total += detailedAnalyze.getProxyFor(relative).getMineProbability();
			}
		}
		return total;
	}

	public static List<MinesweeperField> getBombAdjacents(MinesweeperField field) {
		final int RANGE = 2;
		int xpos = UtilZomisUtils.ensureRange(RANGE, field.getX(), field.getMap().getFieldWidth()  - RANGE - 1);
		int ypos = UtilZomisUtils.ensureRange(RANGE, field.getY(), field.getMap().getFieldHeight() - RANGE - 1);
		field = field.getMap().getPosition(xpos, ypos);
		
		List<MinesweeperField> bombAdjacents = new ArrayList<MinesweeperField>();
		
		int x = field.getX();
		int y = field.getY();
		
		for (int yy = -2 ; yy <= 2 ; yy++) {
			for (int xx = -2 ; xx <= 2 ; xx++) {
				if ((x + xx >= 0 && x + xx < field.getMap().getFieldWidth()) && ( y + yy >= 0 && y + yy < field.getMap().getFieldHeight()))
					bombAdjacents.add(field.getRelativePosition(xx, yy));
			}
		}
		
		return bombAdjacents;
	}
	public static MinesweeperField getBestBomb(MinesweeperMap map, AnalyzeResult<MinesweeperField> analyze) {
		double max = 0;
		MinesweeperField field = null;
		for (MinesweeperField ff : map.getIteration()) {
			double bomb = getBombProbability(getBombAdjacents(ff), analyze);
			
			if (bomb > max) {
				field = ff;
				max = bomb;
			}
		}
		
		return field;
	}
	
	public static double getBombProbability(MinesweeperField bombCenter, AnalyzeResult<MinesweeperField> analyze) {
		return getBombProbability(getBombAdjacents(bombCenter), analyze);
	}
	public static double getBombProbability(Collection<MinesweeperField> affected, AnalyzeResult<MinesweeperField> analyze) {
		
		double total = 0;
		for (MinesweeperField field : affected) {
			if (field.isClicked()) continue;
			FieldGroup<MinesweeperField> know = analyze.getGroupFor(field);
			if (know == null) continue;
			
			total += know.getProbability();
		}
		
		return total;
	}

}
