package net.zomis.minesweeper.scores;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.SortedSet;

import net.zomis.UtilZomisList;
import net.zomis.UtilZomisUtils;
import net.zomis.minesweeper.analyze.detail.ProbabilityKnowledge;
import net.zomis.minesweeper.analyze.impl.AnalyzeProvider;
import net.zomis.minesweeper.game.MinesweeperField;
import net.zomis.minesweeper.game.MinesweeperPlayingPlayer;
import net.zomis.minesweeper.game.model.MapUtils;
import net.zomis.minesweeper.weapons.MinesweeperWeapon;

public class FieldScores implements ScoreParameters {
	private final MinesweeperWeapon	weapon;
	private AnalyzeProvider analyze;
	private final MinesweeperPlayingPlayer	player;
	
	private List<AbstractScorer> activeScorers;
	private final ScoreConfig	config;
	private Map<Class<?>, Object> analyzes;

	private final Map<MinesweeperField, FieldScore> scores = new HashMap<MinesweeperField, FieldScore>();
	private MinesweeperField[][]	ranks;
	
	@SuppressWarnings("unchecked")
	public <E> E getAnalyze(Class<E> clazz) {
//		Logger.getLogger(getClass()).info("Getting analyze for class " + clazz + " size is " + this.analyzes.size());
		return (E) this.analyzes.get(clazz);
	}
	
	FieldScores(MinesweeperPlayingPlayer player, MinesweeperWeapon weapon, ScoreConfig config) {
		this.weapon = weapon;
		this.player = player;
		this.config = config;
	}

	public MinesweeperWeapon getWeapon() {
		return weapon;
	}


	void determineActiveScorers() {
		if (activeScorers != null) throw new IllegalStateException();
		
		activeScorers = new ArrayList<AbstractScorer>();
			
		for (AbstractScorer scorer : config.getScorers().keySet()) {
			if (scorer.workWithWeapon(this)) {
				activeScorers.add(scorer);
			}
		}
	}

	void calculateMoveScores() {
		for (MinesweeperField field : this.player.getMap().getAllFields()) {
			if (!weapon.canUseAt(this.player, field)) continue;
			
			FieldScore fscore = new FieldScore(field);
			ProbabilityKnowledge<MinesweeperField> data = this.analyze == null ? null : this.analyze.getKnowledgeFor(field);
			for (AbstractScorer scorer : activeScorers) {
				double lastScore = scorer.getScoreFor(field, data, this);
				fscore.addScore(scorer, lastScore, this.config.getScorers().get(scorer));
			}
			scores.put(field, fscore);
		}
	}
	
	void calculateRankings() {
		SortedSet<Entry<MinesweeperField, FieldScore>> sorted = UtilZomisList.entriesSortedByValues(scores, true);
			
		int index = 0;
		Double lastValue = null;
		for (Iterator<Entry<MinesweeperField, FieldScore>> it = sorted.iterator(); it.hasNext();) {
			Entry<MinesweeperField, FieldScore> ee = it.next();
			if (lastValue == null || ee.getValue().getScore() != lastValue.doubleValue()) {
				lastValue = ee.getValue().getScore();
				index++;
			}
		}
		int highestIndex = index;

		ranks = new MinesweeperField[highestIndex + 1][];

		index = 0;
		List<MinesweeperField> list = null;
		lastValue = null;
		for (Iterator<Entry<MinesweeperField, FieldScore>> it = sorted.iterator(); it.hasNext();) {
			Entry<MinesweeperField, FieldScore> ee = it.next();
			MinesweeperField field = ee.getKey();
			if (lastValue == null || ee.getValue().getScore() != lastValue.doubleValue()) {
				lastValue = ee.getValue().getScore();
				if (list != null) {
					ranks[index - 1] = list.toArray(new MinesweeperField[list.size()]);
				}
				index++;
				list = new ArrayList<MinesweeperField>();
			}
			list.add(field);
		}
		if (list != null) {
			ranks[index - 1] = list.toArray(new MinesweeperField[list.size()]);
		}
	}

	void postHandle() {
		for (PostScorer post : this.config.getPostScorers()) {
			post.handle(this);
		}
	}

	void setAnalyze(AnalyzeProvider analyze) {
		this.analyze = analyze;
	}

	@Override
	public AnalyzeProvider getAnalyze() {
		return analyze;
	}
	@Override
	public MinesweeperPlayingPlayer getPlayer() {
		return player;
	}

	public MinesweeperField[][] getRankings() {
		return this.ranks;
	}

	public Map<MinesweeperField, FieldScore> getScores() {
		return this.scores;
	}

	public List<FieldScore> getBestFields() {
		return UtilZomisList.getAllExtreme(this.scores.values(), Integer.MIN_VALUE, new _GetValueFieldScore());
	}
	
	private static class _GetValueFieldScore implements UtilZomisList.GetValueInterface<FieldScore> {
		@Override
		public double getValue(FieldScore obj) {
			return obj.getScore();
		}
	}

	public FieldScore getScoreFor(MinesweeperField field) {
		return this.scores.get(field);
	}

	public double getNormalized(MinesweeperField field) {
		if (scores.get(field) == null) throw new IllegalArgumentException("Field does not exist among scores: " + field + " in map: " + MapUtils.saveKnownMap(field.getMap()));
		SortedSet<Entry<MinesweeperField, FieldScore>> sorted = UtilZomisList.entriesSortedByValues(scores, true);
		double max = sorted.first().getValue().getScore();
		double min = sorted.last().getValue().getScore();
		double range = max - min;
		return UtilZomisUtils.normalized(scores.get(field).getScore(), min, range);
	}
	
	public void normalize() {
		
		
		
		
		
	}
	
	
	
	public static boolean listsEquals(List<FieldScore> a, List<FieldScore> b) {
		List<MinesweeperField> temp;
		temp = scoresToFields(a);
		temp.removeAll(b);
		if (!temp.isEmpty()) return false;
		
		temp = scoresToFields(b);
		temp.removeAll(a);
		if (!temp.isEmpty()) return false;
		
		return true;
	}

	public static List<MinesweeperField> scoresToFields(List<FieldScore> scoreList) {
		List<MinesweeperField> list = new ArrayList<MinesweeperField>();
		for (FieldScore score : scoreList) list.add(score.getField());
		return list;
	}

	private List<List<FieldScore>> rankedScores;
	public void rankScores() {
		SortedSet<Entry<MinesweeperField, FieldScore>> sorted = UtilZomisList.entriesSortedByValues(this.scores, true);
		rankedScores = new LinkedList<List<FieldScore>>();
		if (sorted.isEmpty()) {
			
			return;
		}
		double minScore = sorted.last().getValue().getScore();
		double maxScore = sorted.first().getValue().getScore();
		double lastScore = maxScore + 1;
		
//		Logger.getLogger(FieldScores.class).info("Normalizing called, " + lastScore + ", " + minScore + "--" + maxScore + ": " + this, new Exception());
		
		int rank = 0;
		List<FieldScore> currentRank = new LinkedList<FieldScore>();
		for (Entry<MinesweeperField, FieldScore> score : sorted) {
			if (lastScore != score.getValue().getScore()) {
				lastScore = score.getValue().getScore();
				rank++;
				currentRank = new LinkedList<FieldScore>();
				rankedScores.add(currentRank);
			}
			score.getValue().setRank(rank);
			double normalized = UtilZomisUtils.normalized(score.getValue().getScore(), minScore, maxScore - minScore);
//			if (currentRank.size() == 0) Logger.getLogger(FieldScores.class).info("Normalizing " + score.getValue().getScore() + ", " + minScore + "--" + maxScore + ": " + normalized);

			score.getValue().setNormalized(normalized);
			currentRank.add(score.getValue());
		}
	}
	/**
	 * 
	 * @param rank From 1 to getRankLength() 
	 * @return
	 */
	public List<FieldScore> getRank(int rank) {
		return rankedScores.get(rank - 1);
	}

	public int getRankCount() {
		return rankedScores.size();
	}

	public ScoreConfig getConfig() {
		return this.config;
	}

	public void setAnalyzes(Map<Class<?>, Object> analyzes) {
//		Logger.getLogger(getClass()).info("Copying analyzes: " + analyzes + " size " + analyzes.size());
		this.analyzes = new HashMap<Class<?>, Object>(analyzes);
	}
}
